/*****************************************************************/
/*                                                               */
/*                                                               */
/*  File     :     XSCROLL.H                                     */
/*                                                               */
/*  Producer :     Interactive Studio                            */
/*                                                               */
/*        Programmers :  Michal Stencl                           */
/*                                                               */
/*                                                               */
/*  Copyright (c) 2000, Interactive Studio                       */
/*  All Rights Reserved                                          */
/*                                                               */
/*  Contacts :                                                   */
/*                                                               */
/*        Address :  Kozia 29, Bratislava 81103, Slovak Republic */
/*        E-mail  :  developer@InteractiveStudio.sk              */
/*        WEB     :  http://www.InteractiveStudio.sk/            */
/*                                                               */
/*                                                               */
/*  This file is part of QubeOS                                  */
/*                                                               */
/*****************************************************************/

#ifndef __XMENU_H_INCLUDED__
#define __XMENU_H_INCLUDED__

#ifdef __cplusplus
extern "C" {
#endif

#include<matrixos.h>
#include"xscroll.h"

/**
 * Menuview flags
*/
/* Menuview is vertical */
#define MF_VERMENU        0x01
/* Menuview is horizontal */
#define MF_HORMENU        0x02
/* Menuview is alpha blended */
#define MF_ALPHA          0x04

/**
 * Menu action flags
*/
#define MA_NONE             1
#define MA_HMFORWARD       -6
#define MA_OTHER           -5
#define MA_ESC             -4
#define MA_END             -3
#define MA_BACKWARD        -2
#define MA_COMMAND         -1
#define MA_IAMSUBMENU       2
#define MA_SUBMENU          3
#define MA_FORWARD          4
#define MA_ENTER            5


/* messages for copycutdelete */

#define MMSGCCD_COPY        0x0010
#define MMSGCCD_CUT         0x0020
#define MMSGCCD_DELETE      0x0040
#define MMSGCCD_ALL         (MMSGCCD_COPY|MMSGCCD_CUT|MMSGCCD_DELETE)


/* messages for drag and drop */

#define MMSGDD_COPY        0x0010
#define MMSGDD_CUT         0x0020
#define MMSGDD_CANCEL      0x0040
#define MMSGDD_ALL         (MMSGDD_COPY|MMSGDD_CUT|MMSGDD_CANCEL)


/* messages for window type - are used also as flags for this menu */

#define MMSGWT_MINIMIZE     0x0001
#define MMSGWT_MAXIMIZE     0x0002
#define MMSGWT_RESTORE      0x0004
#define MMSGWT_CLOSE        0x0008

/* messages for file drag and drop */

#define MMSGFDD_COPY        0x0010
#define MMSGFDD_CUT         0x0020
#define MMSGFDD_LINK        0x0040
#define MMSGFDD_CANCEL      0x0080
#define MMSGFDD_ALL         (MMSGFDD_COPY|MMSGFDD_CUT|MMSGFDD_LINK|MMSGFDD_CANCEL)


/* messages for file type */

#define MMSGFT_OPEN         11
#define MMSGFT_COPY         12
#define MMSGFT_CUT          0x0800
#define MMSGFT_PASTE        0x0080
#define MMSGFT_MAKELINK     0x0100
#define MMSGFT_DELETE       0x0200
#define MMSGFT_RENAME       0x0400
#define MMSGFT_PROPERTIES   14
#define MMSGFT_ALL         (MMSGFT_RENAME|MMSGFT_DELETE|MMSGFT_PASTE|MMSGFT_MAKELINK|MMSGFT_CUT)


/* messages for filelistbox */

#define MMSGFLB_LARGE_ICONS        11
#define MMSGFLB_LIST               12
#define MMSGFLB_REFRESH            13
#define MMSGFLB_PASTE              0x0080
#define MMSGFLB_PASTELINK          0x0100
#define MMSGFLB_NEW_FOLDER         0x0200
#define MMSGFLB_NEW_SHORTCUT       0x0400
#define MMSGFLB_PROPERTIES         14
#define MMSGFLB_DETAILS            15
#define MMSGFLB_COLLECT_BY_NAME    16
#define MMSGFLB_COLLECT_BY_SIZE    17
#define MMSGFLB_COLLECT_BY_TIME    18
#define MMSGFLB_NEW_THISFOLDER     19
#define MMSGFLB_NEW_DESKTOP        20
#define MMSGFLB_NEW_LOCALHOST      21
#define MMSGFLB_NEW_MYDOCS         22
#define MMSGFLB_ALL                (MMSGFLB_NEW_SHORTCUT|MMSGFLB_NEW_FOLDER|MMSGFLB_PASTELINK|MMSGFLB_PASTE)

/* flags for filelistbox */

#define MFLBF_IS_CHECKED_NAME     0x0001
#define MFLBF_IS_CHECKED_SIZE     0x0002
#define MFLBF_IS_CHECKED_TIME     0x0004
#define MFLBF_IS_CHECKED_LARGE    0x0008
#define MFLBF_IS_CHECKED_LIST     0x0010
#define MFLBF_IS_CHECKED_DETAILS  0x0020


/*
 * Menuitem flags
*/
/* None flags in the item */
#define MIF_NONE      0x00000
/* Item has the same form as checkbox */
#define MIF_CHECK     0x00001
/* Item was checked */
#define MIF_CHECKOK   0x00002
/* Item has own memory of the icon. */
#define MIF_SELFICON  0x00004


/*
  Menu palette. It's set in menu initializing function as the palette
  of the menu. This settings are only default. The real settings are
  read from registers when the xmenu library is load into the memory.

  You can later get these values from your view ( in this case from the
  menu ) by calling my_view->get_color(my_view, @index) where index
  is the index of these values. E.g. my_menu->get_color(my_menu, 0)
  is CO_SYSx20.

  Note that these values are not used. There is [colors] section in
  registers, from where are colors read. See "menu__ini"
  at the bottom of this file for more information.
*/
extern l_color  pal_menu[];

/*
  Menu palette. It's set in menu initializing function as the palette
  of the menu. This settings are only default. The real settings are
  read from registers when the xmenu library is load into the memory.

  You can later get these values from your view ( in this case from the
  menu ) by calling my_view->get_color(my_view, @index) where index
  is the index of these values. E.g. my_menu->get_color(my_menu, 0)
  is CO_SYSx20.

  Note that these values are not used. There is [colors] section in
  registers, from where are colors read. See "menu__ini"
  at the bottom of this file for more information.
*/
extern l_color  pal_hormenu[];

/* Alpha blend of menu. Range from 0..100 */
extern l_int  menu_alpha;

/* t_menu structure */

typedef struct t_menuitem *p_menuitem;
typedef struct t_menu     *p_menu;
typedef struct t_menu {

  /* Points to first item in the menu */
  p_menuitem   items;
  /* Points to last selected item in the menu */
  p_menuitem   current;

} t_menu;


/* t_menuitem structure */

typedef struct t_menuitem {

  /* Points to next item in the menu. if NULL, this is the last item */
  p_menuitem next;
  /* Text of the item */
  l_text     name;
  /* Text that shows info about hotkey of the item on the right side */
  l_text     param;
  /* Keycode of hotkey in this item */
  l_int      hotkey;
  /* Function that's called whenever the item is pressed. The 2nd argument
     is used as on_press_arg, see below. Function returns TRUE if the
     command wads performed */
  l_bool     (*on_press) ( p_menuitem, l_dword );
  /* This value is used as the 2nd argument in function "on_press" */
  l_dword    on_press_arg;
  /* True if item is enable */
  l_bool     enable;
  /* Text to be visible as the info text */
  l_text     info_text;
  /* Flags of the item, see MIF_XXXX above. */
  l_int      flags;
  /* Font of the item */
  p_font     font;
  /* Bitmap as the icon for item. */
  p_bitmap   icon;
  /* Reserved for next versions */
  l_char     reserved[16];
  /* Pointer to sub-menu */
  p_menu     submenu;

} t_menuitem;



/* t_menuview */

typedef struct t_menuview *p_menuview;
typedef struct t_menuview {

  struct t_scroller  obclass;

  p_menu         menu;
/*
  Internal variable
*/
  l_char         action;
/*
  Indicates distance between two items in vertical menu it's (y),
  in horizontal it's (x)
*/
  t_point        between;
/*
  Height of the item
*/
  l_rect         item_size;
/*
  Saved desktop. This desktop is saved whenever is menu showed and released
  when the menu is hiden.
*/
  p_graphics     safe_desktop;
/*
  Sub-menu of this menu
*/
  p_menuview     sub_menu;
/*
  Parent-menu of this menu
*/
  p_menuview     parent_menu;
/*
  Indicates flags of the menu, that can be one of MF_XXXX, see above.
*/
  l_int          flags;
/*
  True if process is inside the menu
*/
  l_bool         inside;

  t_rect         bounds_area;

  void        (*save_desktop) ( p_menuview o );
  l_int       (*at) ( p_menuview o, p_menuitem p );
  p_menuitem  (*index_of) ( p_menuview o, l_int item );
  void        (*draw_item) ( p_menuview o, p_menuitem item );
  void        (*draw_current_item) ( p_menuview o );
  t_rect      (*get_item_rect_ex) ( p_menuview o, p_menuitem item, t_rect r );
  t_rect      (*get_item_rect) ( p_menuview o, p_menuitem item );
  t_rect      (*get_mouseitem_rect) ( p_menuview o, p_menuitem item );
  p_menuitem  (*prev_item) ( p_menuview o, p_menuitem item );
  p_menuitem  (*next_item) ( p_menuview o, p_menuitem item );
  p_menuitem  (*get_next_item) ( p_menuview o, p_menuitem item, l_bool forward );
  l_dword     (*run_new_menu) ( p_menuview o, t_rect r, l_int *action, p_menu menu );
  p_menuitem  (*get_item_from_mouse) ( p_menuview o, t_point p );
  l_bool      (*is_mouse_in_one_of_parent_menu) ( p_menuview o, t_point p );
  l_bool      (*is_mouse_in_one_of_parent_item) ( p_menuview o, t_point p );
  l_bool      (*is_one_of_subs_state) ( p_menuview o, l_dword st );
  p_menuview  (*top_menuview) ( p_menuview o );
  p_menuview  (*is_hor_menu_one_of_parents) ( p_menuview o );
  void        (*blink_item) ( p_menuview o, p_menuitem item );

} t_menuview;

#define MENUVIEW(o)  ((p_menuview)(o))

extern p_menuview def_menuview;
extern p_menuview def_hormenu;

#define DEF_MENUVIEW_FUNC(y) DEF_FUNC(def_menuview,y)
#define DEF_MENUVIEW_SCROLLER_FUNC(y) DEF_FUNC(SCROLLER(def_menuview),y)
#define DEF_MENUVIEW_SCROLLER_VIEW_FUNC(y) DEF_FUNC(VIEW(def_menuview),y)
#define DEF_MENUVIEW_SCROLLER_OBJECT_FUNC(y) DEF_FUNC(OBJECT(def_menuview),y)

#define DEF_HORMENU_FUNC(y) DEF_FUNC(def_hormenu,y)
#define DEF_HORMENU_SCROLLER_FUNC(y) DEF_FUNC(SCROLLER(def_hormenu),y)
#define DEF_HORMENU_SCROLLER_VIEW_FUNC(y) DEF_FUNC(VIEW(def_hormenu),y)
#define DEF_HORMENU_SCROLLER_OBJECT_FUNC(y) DEF_FUNC(OBJECT(def_hormenu),y)

typedef t_menuview t_vermenu;
typedef t_menuview t_hormenu;
typedef t_vermenu *p_vermenu;
typedef t_hormenu *p_hormenu;


/**
 * Adds items of the menu @items at the position @pos to the menu
 * @p. Returns TRUE on success, otherwise FALSE.
*/
l_bool  menu_add ( p_menu p, l_int pos, p_menuitem items );


/**
 * Makes new menu that contains @item as the pointer to first @item
 * in the menu.
*/
p_menu  menu_new ( p_menuitem item );


/**
 * Makes a menuitem that will be shown as horizontal line in the menu.
*/
p_menuitem  menu_new_line ( p_menuitem next );


/**
 * Makes new menuitem that will show the sub-menuitem, where
 * @name    - is the title of the item
 * @enable  - indicates if the item is enabled or disabled.
 * @font    - sets the font of the item, if NULL the standard font is
 *            used.
 * @icon    - bitmap of the icon that's used for sub-menu. If
 *            set as NULL, the default icon is used.
 * @menu    - the pointer to sub-menu.
 * @next    - pointer to next menu item.
*/
p_menuitem  menu_new_sub2 ( l_text name, l_bool enable, p_font font,
                            p_bitmap icon, p_menu menu, p_menuitem next );


/**
 * Makes new menuitem that will show the sub-menuitem, where
 * @name    - is the title of the item
 * @menu    - the pointer to sub-menu.
 * @next    - pointer to next menu item.
*/
p_menuitem  menu_new_sub  ( l_text name, p_menu menu, p_menuitem next );


/**
 * Releases all of the memory previously allocated by function
 * "menu_new_item".
*/
void   menu_free_item ( p_menuitem i );


/**
 * Releases all the memory of menu @i and its sub-menus and items.
*/
void   menu_free ( p_menu i );


/**
 * Makes the new item of the menu as the command item, where
 * @name    - title of menuitem
 * @param   - text for params that's used as the info of hotkeys.
 * @hotkey  - hotkey of menuitem, must be in value of keycode.
 * @enable  - makes item enable if TRUE, or disable if FALSE
 * @flags   - flags of the item, that may be the combination of followings :
 *
 *            MIF_CHECK    - menuitem is used as the checkbox
 *            MIF_CHECKOK  - only used if the flag above is used and
 *                           indicates, that the item is checked
 *            MIF_SELFICON - this flag is used only if @icon exists and
 *                           we want to destroy context of the icon when
 *                           menuitem will be released. It can't be used
 *                           in global icons or in the case you want to
 *                           use this icon later. This indicates, that the
 *                           menu can do with @icon anything it wants
 *                           and you may be sure you will never use it.
 *
 * @font    - font of menuitem. If it's NULL, it's used default font.
 * @icon    - bitmap of item. See @flag MIF_SELFICON.
 * @on_press - the function that's called when the items is pressed. This
 *             function is called with 2 arguments. The first is this item
 *             and the second one is the same argument as you set in
 *             @on_press_arg, see later.
 * @on_press_arg - this argument is used as the 2nd argument in function
 *                 @on_press, when the item is pressed. May be NULL. It's
 *                 your argument, you can use free.
 * @next    - pointer to next menu item.
*/
p_menuitem  menu_new_item2 ( l_text name, l_text param, l_int hotkey,
                             l_bool enable, l_int flags,
                             p_font font, p_bitmap icon,
                             l_bool (*on_press) ( p_menuitem o, l_dword on_press_arg ),
                             l_dword on_press_arg,
                             p_menuitem next );


/**
 * Makes the new item of the menu as the command item, where
 * @name    - title of menuitem
 * @on_press - the function that's called when the items is pressed. This
 *             function is called with 2 arguments. The first is this item
 *             and the second one is the same argument as you set in
 *             @on_press_arg, see later.
 * @on_press_arg - this argument is used as the 2nd argument in function
 *                 @on_press, when the item is pressed. May be NULL. It's
 *                 your argument, you can use free.
 * @next    - pointer to next menu item.
*/
p_menuitem  menu_new_item ( l_text name,
                            l_bool (*on_press) ( p_menuitem o, l_dword on_press_arg ),
                            l_dword on_press_arg,
                            p_menuitem next );


/**
 * Returns position of item @x in menu @m. If the item @x not exists
 * in the menu, -1 is returned.
*/
l_int  menu_at ( p_menu m, p_menuitem x );

/**
 * Returns number of items in the menu @m, or ZERO if there is no item.
*/
l_int  menu_get_max_item ( p_menu m );


/**
 * Returns an item of the menu @p, that's have
 * the same hotkey as @keycode, or NULL.
*/
p_menuitem  menu_get_by_hotkey ( p_menu m, l_int keycode );


/**
 * Returns an item of the menu @p, what a title is the @name disregarding
 * case.
*/
p_menuitem  menu_get_by_name ( p_menu m, l_text name );


/**
 * Sets enable or disable menu item that has the same title as @name and
 * returns TRUE on success.
*/
l_bool menu_set_enable ( p_menu m, l_text name, l_bool set );


/**
 * Gets the correct with and height of the menu @menu, where @h is the
 * height of one item.
*/
t_point  menu_vertical_get_size ( p_menu menu, l_int h );


/**
 * Executes @i item of the menu and returns TRUE on success.
*/
l_bool menu_execute_item ( p_menuitem i );


/**
 * Makes menu modal and returns p_menuitem as the pointer to
 * t_menuitem of an item that was pressed. If no item was pressed,
 * function returnd NULL. Returned value is only for information, because
 * "on_press" function of the pressed item is called inside.
*/
p_menuitem menu_run ( p_menuview o );


/**
 * Creates new vertical menuview at the point @p and information about
 * menu are stored in @menu. See "menu_new" for more info.
*/
p_menuview  vermenu_new ( t_point p, p_menu menu );

/**
 * Creates new vertical menuview at the point @p and information about
 * menu are stored in @menu. See "menu_new" for more info.
 *
 * @l - limits of owners bounds where can be menu visibled
*/
p_menuview  vermenu_new2 ( t_point p, t_rect l, p_menu menu );

/**
 * Initializes new menuview as the parent of all styles of menu, where
 * @r - is the area where menu will be placed. It's irelevant, only r.a
 *      is used as the left,top point. The size of menu is calculating in
 *      the function.
 *
 * @l - limits of owners bounds where can be menu visibled
 * @flags - flags that indicates style of menu : MF_VERMENU, MF_HORMENU
 * @item_size - is the height of item in the menu
 * @menu      - information about menu. See "menu_new" for more info.
*/
extern p_menuview  (*menuview_init2) ( p_menuview o, t_rect r, t_rect l, l_int flags, l_rect item_size,
                                p_menu menu );
p_menuview  _menuview_init2 ( p_menuview o, t_rect r, t_rect l, l_int flags, l_rect item_size,
                              p_menu menu );


/**
 * Creates new horizontal menuview at the rect @r and information about
 * menu are stored in @menu. See "menu_new" for more info.
*/
p_menuview  hormenu_new ( t_rect p, p_menu menu );

/**
 * Initializes new horizontal as the parent of all styles of menu, where
 * @r - is the area where menu will be placed. It's irelevant, only r.a
 *      is used as the left,top point. The size of menu is calculating in
 *      the function.
 * @menu      - information about menu. See "menu_new" for more info.
*/
extern p_hormenu  (*hormenu_init) ( p_menuview o, t_rect r, p_menu menu );
p_hormenu  _hormenu_init ( p_menuview o, t_rect r, p_menu menu );



/**
 * Menu types...
*/

/**
 * Used in drag and drop, where @add is pointer to items we want to add.
*/
p_menu  menutype_dragdrop ( l_int flags, p_menuitem add );

/**
 * Used in window icon, where @add is pointer to items we want to add.
*/
p_menu  menutype_window ( l_int flags, p_menuitem add );

/**
 * Used for fileicon menu
*/
p_menu  menutype_file ( l_int flags, p_menuitem add );

/**
 * Used for copy/cut
*/
p_menu  menutype_copycutdelete ( l_int flags, p_menuitem add );

/**
 * Used for filelistbox
*/
p_menu  menutype_filelistbox ( l_int flags, p_menuitem add );

/**
 * Used in file drag and drop, where @add is pointer to items we want to add.
*/
p_menu  menutype_filedragdrop ( l_int flags, p_menuitem add );


#ifdef __cplusplus
} /* end of poor C code */
#endif

#endif /* end of xmenu.h */

