/*****************************************************************/
/*                                                               */
/*                                                               */
/*  File     :     XCOORD.H                                      */
/*                                                               */
/*  Producer :     Interactive Studio                            */
/*                                                               */
/*        Programmers :  Michal Stencl                           */
/*                                                               */
/*                                                               */
/*  Copyright (c) 2000, Interactive Studio                       */
/*  All Rights Reserved                                          */
/*                                                               */
/*  Contacts :                                                   */
/*                                                               */
/*        Address :  Kozia 29, Bratislava 81103, Slovak Republic */
/*        E-mail  :  developer@InteractiveStudio.sk              */
/*        WEB     :  http://www.InteractiveStudio.sk/            */
/*                                                               */
/*                                                               */
/*  This file is part of QubeOS                                  */
/*                                                               */
/*****************************************************************/

/*
  First release : 06.09.2000 by Michal Stencl
*/

#ifndef __XCOORD_H_INCLUDED__
#define __XCOORD_H_INCLUDED__

#ifdef __cplusplus
extern "C" { /* begin poor C code */
#endif

#include"basics.h"
#include"drvbas.h"


/*
  t_point structure
*/
typedef struct t_point *p_point;
typedef struct t_point {

  l_rect    x;
  l_rect    y;

} t_point;


/*
  t_relpoint structure
*/
typedef struct t_relpoint *p_relpoint;
typedef struct t_relpoint {

  l_relrect    x;
  l_relrect    y;

} t_relpoint;


/*
  rectangle structure
*/
typedef struct t_rect *p_rect;
typedef struct t_rect {

  struct t_point   a;
  struct t_point   b;

} t_rect;

extern t_rect rect_empty;



/*
  Creates the new t_point of values @x, @y
*/
t_point point_assign ( l_rect x, l_rect y );


/*
  Creates the new t_point of values @x, @y. The same as point_assign, but
  this returns the chunk of memory that must be later released by caller.
*/
p_point point_new ( l_rect x, l_rect y );


/*
  Creates the new t_relpoint of values @x, @y
*/
t_relpoint relpoint_assign ( l_relrect x, l_relrect y );


/*
  Returns fixed points from relatives.
  Relative l_relrect is in the range from <0,1>
*/
t_point relpoint_fixpoint ( t_relpoint rel, t_point fix );


/*
  Creates the t_rect of values @ax, @ay, @bx, @by
*/
t_rect  rect_assign ( l_rect ax, l_rect ay, l_rect bx, l_rect by );


/*
  Creates the new p_rect. The same as rect_assign, but this returns
  the chunk of memory that must be released by the caller.
*/
p_rect  rect_new ( l_rect ax, l_rect ay, l_rect bx, l_rect by );


/*
   Moves the coordination of the rectangle t_rect @r of delta @mx in
   x coord and delta @my in y coord.
*/
t_rect  rect_move ( t_rect r, l_rect mx, l_rect my );


/*
  Returns the size of rectangle. Returns width in t_point.x and height
  in t_point.y
*/
t_point rect_size ( t_rect r );


/**
 * Corrects rectangle if r.a.x is greater than r.b.x or r.a.y is greater than
 * r.b.y. Returns correct recangle.
*/
t_rect  rect_correct ( t_rect r );


/*
  Interesects the rectangle and returns the new one, where rectangle
  @r overlay rectangle @d. If they not overlay, it returns @rect_empty
*/
t_rect  rect_cliped ( t_rect r, t_rect d );


/*
   Tests if rectangle @d overlay rectangle @r. If it so, returns true,
   otherwise returns false.
*/
l_bool  rect_overlay ( t_rect r, t_rect d );


/*
  Returns the width of the rectangle in l_long

  Note :
  l_rect is the l_long values not the same to t_rect what's the structure
*/
l_rect rect_sizex ( t_rect r );


/*
  Returns the height of the rectangle in l_long

  Note :
  l_rect is the l_long values not the same to t_rect what's the structure
*/
l_rect rect_sizey ( t_rect r );


/*
  Tests if the rectangle @r is empty. If it so, returns true,
  otherwise returns false
*/
l_bool rect_check_empty ( t_rect r );


/*
   Tests if the point @p is placed inside the rectangle @r. If it so,
   returns true, otherwise returns false.
*/
l_bool rect_contains ( t_rect r, t_point p );


/*
   Tests if rectangle @r is the same as rectangle @nr. If it so, returns
   true, otherwise returns false.
*/
l_bool  rect_equals ( t_rect r, t_rect nr );


/*

*/
void  rect_double_overlay ( t_rect* fr, t_rect *lr );


/**
 * Returns 4 rectangles as the result of @out - @in, where the 0th
 * t_rect is the left side, 1st t_rect is the right side, 2nd t_rect is
 * the top side, 3rd t_rect is thebottom side and the last one is rect_empty.
 * If rectangle @in is not placed inside the @out, returns NULL.
 *
 *  @out
 * |-----------------|
 * |   |    2    |   |
 * |   |---------|   |
 * |   |XXXXXXXXX|   |
 * | 0 |XXX@inXXX| 1 |
 * |   |---------|   |
 * |   |    3    |   |
 * |-----------------|
 *
*/
p_rect  rect_inside_overlay ( t_rect out, t_rect in  );


/* first in first out */

#define fifo_add_rect(l,s,w) ((p_rect)fifo_add((l_ptr)l, (l_ptr)s, sizeof(t_rect), w))
#define fifo_get_rect(l,w)   ((p_rect)fifo_get((l_ptr*)l, sizeof(t_rect), w))


#ifdef __cplusplus
} /* end of poor C code */
#endif

#endif /* end of xcoord.h */
