/*****************************************************************/
/*                                                               */
/*                                                               */
/*  File     :     XFILE.H                                       */
/*                                                               */
/*  Producer :     Interactive Studio                            */
/*                                                               */
/*        Programmers :  Michal Stencl                           */
/*                                                               */
/*                                                               */
/*  Copyright (c) 2000, Interactive Studio                       */
/*  All Rights Reserved                                          */
/*                                                               */
/*  Contacts :                                                   */
/*                                                               */
/*        Address :  Kozia 29, Bratislava 81103, Slovak Republic */
/*        E-mail  :  developer@InteractiveStudio.sk              */
/*        WEB     :  http://www.InteractiveStudio.sk/            */
/*                                                               */
/*                                                               */
/*  This file is part of MatrixOS                                */
/*                                                               */
/*****************************************************************/

/*
  First release : 06.09.2000 by Michal Stencl
*/

#ifndef __XFILE_H_INCLUDED__
#define __XFILE_H_INCLUDED__

#ifdef __cplusplus
extern "C" { /* begin poor C code */
#endif

#include"basics.h"
#include"drvbas.h"

/*
  File system of this OS is case sensitive for all users.
  Only programmer, who programs protocols should know how the
  background works.
*/


/* error codes for file->error */
#define F_ERR_CANT_OPEN_FILE             -1000
#define F_ERR_FILE_IS_LOCK_FOR_ALL       -1001
#define F_ERR_FILE_IS_LOCK_FOR_WRITING   -1002


/* flags of t_file structure */

#define  F_OBJECT              0x001000
#define  F_TEMP                0x002000
#define  F_CANCELEDALL         0x004000


/* used in file_list flags */
/*
  F_TREELIST is used when we want to list also subdirectories
*/
#define  F_TREELIST      0x0001
/*
  F_STOREVECTOR is used when we want store the vector of listed
  files/directrories
*/
#define  F_STOREVECTOR   0x0002
/*
  F_COPY used only in copy style of listing. So the callback is NOT
  called for directories. Copying makes directory automatically by
  _file_copy. Please use this flag only when don't want to call callback
  for directories.

  NOT use in deleting and renaming, because directories will be not deleted
*/
#define  F_COPY          0x0004

/**
 * Listing of the directory will include parent directory as well.
*/
#define  F_INCLUDEPARENT 0x0008

/**
 * file_list will returns in *length argument number of files
*/
#define  F_NUMBERS       0x0010



/* used in file->log

   @obj - object that control this overwriting
   @warning - this info - FL_DELETING
   @o       - file that's deleting
   @info    - NULL
   @info2   - NULL
*/
#define  FL_DELETING    1

/* used in file->log

   @obj - object that control this overwriting
   @warning - this info - FL_OVERWRITING
   @o       - file that's overwriting from
   @info    - file that's overwriting to
   @info2   - NULL
*/
#define  FL_OVERWRITING 2

/* used in file->log

   @obj - object that control this tranfering
   @warning - this info - FL_TRANSFERING
   @o       - file that's transfering from
   @info    - file that's transfering to
   @info2   - size of current transfer
*/
#define  FL_TRANSFERING 3

/* used in file->log

   @obj - object that control this directory making
   @warning - this info - FL_MKDIR
   @o       - directory that's making
   @info    - NULL
   @info2   - NULL
*/
#define  FL_MKDIR       4

/* used in file->log

   @obj - object that control starting of tranfering
   @warning - this info - FL_BEGINTRANSFERING
   @o       - file that's transfering from
   @info    - file that's transfering to
   @info2   - NULL
*/
#define  FL_BEGINTRANSFERING 5


/* used in file->log

   @obj - object that control this tranfering
   @warning - this info - FL_READING
   @o       - file that's reading
   @info    - NULL
   @info2   - size of current transfer
*/
#define  FL_READING        6


/* used in file->log

   @obj - object that control this tranfering
   @warning - this info - FL_WRITING
   @o       - file that's writing to
   @info    - NULL
   @info2   - size of current transfer
*/
#define  FL_WRITING        7

/* used in file->log

   @obj - object that control this renaming
   @warning - this info - FL_BEGINRENAMING
   @o       - file that's renaming from
   @info    - file that's renaming to
   @info2   - NULL
*/
#define  FL_BEGINRENAMING  8

/* used in file->log

   @obj - object that control this operation
   @warning - this info - FL_FILELOCKED_READING
   @o       - file that's can't be read
   @info    - NULL
   @info2   - NULL
*/
#define FL_FILELOCKED_READING   9

/* used in file->log

   @obj - object that control this operation
   @warning - this info - FL_FILELOCKED_REMOVING
   @o       - file that's can't be removed
   @info    - NULL
   @info2   - NULL
*/
#define FL_FILELOCKED_REMOVING  10

/* used in file->log

   @obj - object that control this operation
   @warning - this info - FL_FILELOCKED_WRITING
   @o       - file that's can't be written
   @info    - NULL
   @info2   - NULL
*/
#define FL_FILELOCKED_WRITING   11

/* used in file->log

   @obj - object that control this operation
   @warning - this info - FL_FILELOCKED_MAKING
   @o       - file that's can't be made
   @info    - NULL
   @info2   - NULL
*/
#define FL_FILELOCKED_MAKING    12

/* used in file->log

   @obj - object that control this operation
   @warning - this info - FL_FILELOCKED_RENAMING
   @o       - file that's can't be renamed
   @info    - NULL
   @info2   - NULL
*/
#define FL_FILELOCKED_RENAMING  13


#define  FL_OK            0
#define  FL_CANCEL     -100
#define  FL_CANCELALL  -101

/**
 * Safe information about link to file. It's the same structure as the
 * *.lnk files have.
*/
typedef struct t_filelink *p_filelink;
typedef struct t_filelink {

  l_text       linkto_path;
  l_char       reserved[48];

} t_filelink;


/*
  Functions file_listXXX return vector that contains in the each of the
  element this structure.
*/

typedef struct t_filelistresult *p_filelistresult;
typedef struct t_filelistresult {

  l_word       attrib; /* attrib of the file */
  l_time       mtime;  /* modification time of the file */
  l_time       ctime;  /* creation time of the file */
  l_time       atime;  /* accessing time of the file */
  l_long       size;   /* size of the file or the size of all directory */
  l_text       name;   /* name of the file */
  l_text       path;   /* path to the file */
  l_int        level;  /* level in the sub-directories from 0-n */

} t_filelistresult;


typedef struct t_file *p_file;
typedef struct t_file {

/*
   Pointer to protocol that's used for all files primitives. See
   xprotocol.h for more information. Computer drives are supported by
   localhost protocol.
*/
   p_protocol  protocol;
/*
*/
   l_long     flags;

/*
*/
   l_long     locked_removing;
/*
*/
   l_long     locked_writing;
/*
*/
   l_long     locked_reading;

/*
   path of the file or directory
*/
   l_text     path;

/*
   if flags contains F_OBJECT, then all functions that can take
   more time and are available to stop by some "button" check this
   object each cycle if still exists. If not, process is stoped.
*/
   l_ptr      o;

/*
   While some functions returns pointer or NULL in the case, that something
   was wrong and there may be 2 or more choises of wrong processing, this
   error is used for selecting what error was occured.
*/
   l_long     error;

/*
   Opened file.
*/
   l_protocolfile  opened;

/*
   call this function everytime when something happend. e.g. file exists
   and we are going to replace it. So this function is called by
   @flag FL_XXXX, @o is @this and @info is p_file of the new file,
   that going to replace the old one...
   If this function returns FL_CANCEL or FL_CANCELALL - this function must
   stop the process. All cancels warning are begining from  -100..
*/
   l_long (*log) ( l_ptr obj, l_long warning, p_file o, l_ptr info, l_ptr info2 );


} t_file;


void  file_lock_all ( p_file o );

void  file_unlock_all ( p_file o );

/*
  zero if not-locked, non-zero if locked.
*/
l_long  file_is_lock_all ( p_file o );


/*
  Locks the file. If the file is locked, it can't be open by F_WRITE or
  removed from the system. Reading is available. When file is locked it means
  something is reading or writing from/to this file.

  When you will write it again, you can rewrite important data.

  If file is lock for writing, it's also locked for removing. Not call again
  locking for removing.
*/
void  file_lock_writing ( p_file o );


/*
  Returns pointer where data of the file are stored. Usually NULL for
  media, disks. Not NULL in the case of memory. See xmemory.h

  It's used in the case of the protocol can manage only
  one file. Made for compatibility of memory protocol with others.

  Please be carefull, because the pointer returned by this function
  can be used only while file @o is open. When file is closed, this pointer
  is freed, so if you want to use it later, store it to another memory
  block.
*/
l_ptr   file_linear ( p_file o );


/*
  Locks the file for removing. If the file was locked, this file can't be
  removed. Locks before reading/writing.
*/
void  file_lock_removing ( p_file o );

/*
  Unlock file that was locked before, and returns number of locking at the
  time ( after unlocking ). Each file can be locked more times.

  Returns how many times is still locked for reading.
*/
l_long  file_unlock_writing ( p_file o );

/*
  Unlock file for removing. If file was unlocked absolutly, this file can be
  ereased.
*/
l_long  file_unlock_removing ( p_file o );


/*
  Returns how many times was file locked for writing
*/
l_long  file_is_lock_writing ( p_file o );

/*
  Returns how many times was file locked for removing
*/
l_long  file_is_lock_removing ( p_file o );


/*
  Tests if application can access to the file by the flags

  F_EXISTS     - file exists
  F_DIRECTORY  - file is directory
  F_ARCHIVE    - file is archived file
  F_READABLE   - file can be read
  F_WRITEABLE  - file can be wrote
  F_EXECUTABLE - file can be executed
*/
l_int  file_access ( p_file o, l_int flags );


/*
  Tests if application can read from the specified file
*/
l_bool file_can_read ( p_file o );


/*
  Tests if application can write to the specified file
*/
l_bool file_can_write ( p_file o );

/*
  Tests if the specified file is directory
*/
l_bool file_is_directory ( p_file o );


/*
  Tests if the specified file is "normal" file
*/
l_bool file_is_file ( p_file o );


/*
  Change atributes of the file
*/
l_bool file_chmod ( p_file o, l_int flags );


/*
  Returns content-type of the file. This is independent way to get what type
  the file is. By this way can get e.g. extension of the file in MS-DOS.
*/
l_text file_content_type ( p_file o );


/*
   Tests if content of c1 equals to content of c2
*/
l_bool file_content_equals ( l_text c1, l_text c2 );


/*
   Tests if the name of any part of the path equals to any part of the
   path o2.

   Returns the same values as string_compare or string_icompare.

   You can use also string_copmare.
*/
l_int file_compare ( l_text o, l_text o2 );


/*
   Tests if file @o equals to @o2. The same like !file_compare, only
   use p_file.
*/
l_bool file_equals ( p_file o, p_file o2 );


/*
   Returns the name of the file in MFS1 format
*/
l_text file_get_name ( p_file o );


/**
 * Returns the name without content type
*/
l_text file_get_name_only ( p_file o );


/*
   Returns the parent part of the pathname of this p_file object or
   null if the name has no parent part
*/
l_text _file_get_parent ( p_file o );


/*
  Add file/directory to old file/directory and returns the new file/directory
*/
p_file  file_add_path ( p_file o, l_text pathadd );


/*
   Gets parent path and returns a new filem that point to this path
*/
p_file  file_get_parent ( p_file o );

/*
   Change directory of the file to the new one
*/
l_bool  file_change_path ( p_file o, l_text pathnew );


/**
 * Sets handle object for t_file structure, that takes control over
 * t_file.
*/
l_bool  file_set_handle_object ( p_file o, l_ptr obj );


/**
 * Sets "log" function for t_file structure
*/
l_bool  file_set_logfunc (  p_file o, l_long (*log) ( l_ptr obj, l_long warning, p_file o, l_ptr info, l_ptr info2 ) );


/**
 * Returns true if file is placed in path "localhost://"
*/
l_bool file_is_sysdep ( p_file f );


/*
   Change directory of the file to parent directory.
   The old file maybe also "normal" file. It goes to parent
   direcotry and sets all settings that are important for directory
*/
l_bool  file_go_parent ( p_file o );

/*
   Returns the path of the file in MFS1 format
*/
l_text file_get_path ( p_file o );


/*
  Please use this function when you open it by file_temp, otherwise temporary
  will still exists.
*/
p_file file_free_temporary ( p_file o );


/*
   Returns the absolute path of the file in MFS1 format
*/
l_text file_get_absolute ( p_file o );


/*
  Break up the path to all subdirectories that path contains.
  Returns the vector (list) of Strings where each subdir represents
  one element in the vector. The list start from right to left, what means :
  localhost://dev/c/mydocs/
  0. localhost:
  1. dev
  2. c
  3. mydocs
*/
p_vector file_smash ( p_file o );


/*
  Makes one directory by the same protocol as defined in p_file(o)
  If directory was succesfull made returns true, otherwise it returns false.

  file_mkdir(o, "localhost://dev/c/matrixos");

  makes directory martrixos in directory
  get_property(system_path)+link to directory defined in file (c)
*/
l_bool  file_mkdir ( p_file o, l_text dir );

l_int   file_get_stamp(p_file o, t_filestamp *t );
l_int   file_set_stamp (p_file o, cl_text path, t_filestamp t );
l_time  file_get_last_modified ( p_file o );


/**
 * Returns true if files in vector @src can be copied to file @p.
*/
l_bool file_can_copy ( p_file dst, p_vector src );


/**
 * This function erase the same file names from @dst that equals to
 * @src files.
*/
p_vector file_correct_same ( p_vector dst, p_vector src );


/**
 * Returns new vector of p_file's where the names of file from @src
 * are converted to @dst path + this name.
*/
p_vector file_correct_path ( p_file dst, p_vector src );


/*
  Returns path that is the result of path+'/'+file. This also fix the
  path.

  E.g. path = "localhost://dev/matrix/"
       file = "matrix.txt"

  The result is : "localhost://dev/matrix/matrix.txt" that replaced
  memory of @path what was increased by length equals to string_len(matrix.txt)
  and the new memory is returned.

  E.g. path = "localhost://dev/matrix" - without '/'
       file = "matrix.txt"

  The result is the same as above, only @path memory was replaced and increased
  by length equals to sizeof(l_char)+string_len(matrix.txt).

*/

l_text  file_concate_path ( l_text path, l_text file );


/**
 * Returns true if the path @f is the top directory.
*/
l_bool  file_is_top ( l_text path );


/**
 * Concate path @path with new content type @ct.
*/
l_text  file_concate_content ( l_text path, l_text ct );


/*
 get unique name of file that not exists in path.
 Returns the new name of this file.
*/
l_text  file_unique ( p_file o, l_text prefix, l_text sufix );


/*
  Find all files/directories placed in path of the file and returns
  vector to p_filelistresult.
  @d        - when used this flag, callback will call with 2 parameters
  @size     - the size of the file/directory
  @ext      - what files to find (*.txt - all text files will looking for )
  @attrib   - attribs of files we want to looking for
  @flags    - if want to looking in subdirectories set F_TREELIST
            - if want to add to vectors set F_STOREVECTOR
  @filter   - function, called for each file/directory, where @d is the
              t_file of file that was currently found
  @callback - function called for each file and for each directory as well
              with o->path argument. If file is subdirectory
              this function is called after loop of this subdirectory.
*/
p_vector file_list ( p_file o, p_file d, l_long *size, l_text ext,
                     l_int attrib, l_int flags, l_bool (*filter)(p_file d),
                     l_int (callback)() );


/**
 * Collects vector of p_findresult that was return by function file_list
 * by name. The same vector is returned but the items inside are collected
 * by the name.
*/
p_vector file_collect_by_name ( p_vector listresult );


/**
 * Collects vector of p_findresult that was return by function file_list
 * by time. The same vector is returned but the items inside are collected
 * by the time.
*/
p_vector file_collect_by_time ( p_vector listresult );


/**
 * Collects vector of p_findresult that was return by function file_list
 * by size. The same vector is returned but the items inside are collected
 * by the size.
*/
p_vector file_collect_by_size ( p_vector listresult );


/*
  Returns number of files or directories. If @tree is set to
  true, returns size included subdirectories
*/
l_long file_howmuch ( p_file o, l_bool tree );


/*
  Make all directories in the path of dir. Directories will be made
  by the same protocol as in the file (o). Returns the number of directories
  that were succesfull made.
*/

l_int  file_mkdirs ( p_file o );


/**
 * Returns prefix of protocol of file @o
*/
l_text  file_get_protocol ( p_file o );


/*
  Returns size of the file or directory. If @tree is set to
  true, returns size included subdirectories
*/
l_long file_length ( p_file o, l_bool tree );


/*
   Removes the specified file/directory and returns vector if
   flags was set to F_STOREVECTOR, otherwise NULL
   @o      -  file/directory to remove
   @ext    -  in the case the file is directory, remove all files that have same
              style of extension. If NULL all files will be removed
   @size   -  where will be returned number of files+directories successfull removed
   @attrib -  what attribution of files will be supported for removing
   @flags  -  combination of F_TREELIST or F_STOREVECTOR
   @filter -  function that filters what type of file can be removed. Called
              for each file. May be NULL.
*/
p_vector file_remove2 ( p_file o, l_text ext, l_long *size, l_int attrib, l_int flags, l_bool (*filter)(p_file d) );


/*
   Renames the specified file/directory and returns vector if
   flags was set to F_STOREVECTOR, otherwise NULL
   @o - is the source of the file/directory that will be renamed
   @d - is the destionation file/directory
   @ext    -  in the case the file is directory, rename all files that have same
              style of extension. If NULL all files will be renamed
   @size   -  where will be returned number of files+directories successfull renamed
   @attrib -  what attribution of files will be supported for renaming
   @flags  -  combination of F_TREELIST or F_STOREVECTOR
   @filter -  function that filters what type of file can be renamed. Called
              for each file. May be NULL.

   NOTE : This function use protocol->rename function only if the path
          of source and destination file is the same. Otherwise use copy
          and delete algorithmus.
*/
p_vector  file_rename2 ( p_file o, p_file d, l_long *size, l_text ext, l_int attrib, l_int flags, l_bool (*filter)(p_file d)  );

/*
   Removes the specified file/directory and returns number of
   files that were looking for.
*/
l_long file_remove ( p_file o );


/*
   Copies the specified file/directory and returns vector if
   flags was set to F_STOREVECTOR, otherwise NULL
   @o - is the source of the file/directory that will be copying
   @d - is the destionation file/directory
   @ext    -  in the case the file is directory, copy all files that have same
              style of extension. If NULL all files will be removed
   @size   -  where will be returned number of files+directories successfull copyied
   @attrib -  what attribution of files will be supported for copying
   @flags  -  combination of F_TREELIST or F_STOREVECTOR
   @filter -  function that filters what type of file can be copyied. Called
              for each file. May be NULL.

   NOTE: When you want to copy file or directory you must set the
   real source file/directory and destination file/directory where you
   want to copy the file or directory - NOT *ONLY* the destination path.
   E.g. :

   BAD EXAMPLE !

   p_file o = file_new("localhost://dev/c/matrix/matrixos.txt");
   p_file d = file_new("localhost://dev/c"); !!!! - BAD - only destination path
   file_copy2(o, d, ....)

   GOOD EXAMPLE !

   p_file o = file_new("localhost://dev/c/matrix/matrixos.txt");
   p_file d = file_new("localhost://dev/c/matrixos.txt"); GOOD
   file_copy2(o, d, ....)
*/
p_vector  file_copy2 ( p_file o, p_file d, l_long *size, l_text ext, l_int attrib, l_int flags, l_bool (*filter)(p_file d)  );


/*
   Copies the specified file/directory and returns number of
   files that were looking for.
   o - is the source of the file/directory that will be copying
   d - is the destionation file/directory


   NOTE: When you want to copy file or directory you must set the
   real source file/directory and destination file/directory where you
   want to copy the file or directory - NOT *ONLY* the destination path.
   E.g. :

   BAD EXAMPLE !

   p_file o = file_new("localhost://dev/c/matrix/matrixos.txt");
   p_file d = file_new("localhost://dev/c"); !!!! - BAD - only destination path
   file_copy2(o, d, ....)

   GOOD EXAMPLE !

   p_file o = file_new("localhost://dev/c/matrix/matrixos.txt");
   p_file d = file_new("localhost://dev/c/matrixos.txt"); GOOD
   file_copy2(o, d, ....)
*/
l_long file_copy ( p_file o, p_file d );


/*
   Copies the specified file/directory and returns number of
   files that were looking for.
   o - is the source of the file/directory that will be renamed
   d - is the destionation file/directory
*/
l_long  file_rename ( p_file o, p_file n );


/*
   Add source file from the current position at the current position
   of destination.

   @maxlen - amount of data that will be copied. In the case maxlen is -1,
             the whole source from the current position is copied to the
             destination one.

   Return size of data sucesfull copied
*/
l_long file_add ( p_file o, p_file d, l_long maxlen );


/*
  Removes @size of bytes from the file "file" from position @from.
  This file must be opened by function file_open before calling this func.
  Returns size of bytes that was succesfull removed from the file.
  NOTE: Please use this function for big chunk of data...not for each byte.
*/
l_long file_delete_in ( p_file o, l_long from, l_long size );

/*
  Inserts (ptr) to the file. Please insert more chunk of data because
  this function is slow. It must read all data from the file, then inserts
  @ptr to data that was succesfull read, then makes a temporary file, copies
  the new data of the file to temporary one and then recopy temporary file to
  the old one.

  For more security there is used temporary file where new data are
  firstly copied and in the case that everything was succesfull,
  rewrites the old file by the temporary one.
*/
l_long file_insert_in ( p_file o, l_ptr ptr, l_long from, l_long size );

l_long file_write_num ( p_file o, l_ptr p, l_long size );
l_long file_read_num ( p_file o, l_ptr p, l_long size );
l_long file_write ( p_file o, l_ptr p, l_long size );
l_long file_read ( p_file o, l_ptr p, l_long size );
l_long file_get_pos ( p_file o );
l_int  file_set_pos ( p_file o, l_long pos );

/*
   Writes the string to the file.
   @ptr - text that will be stored in the file ended by '\0' if
          file was open by F_BINARY flag

   Returns the size of bytes succesfull read. Please note that this size
   is not the size of string in the case of F_BINARY,
   but by default the size of string+4+1.

   This function firstly copy the length of text +sizeof(l_char), then
   copy the string of the size "size+sizeof(l_char)".
*/
l_long file_write_string ( p_file o, l_text str );
l_long file_write_int ( p_file o, l_int i );
l_long file_write_short ( p_file o, l_short i );
l_long file_write_long ( p_file o, l_long i );
l_long file_write_float ( p_file o, l_float i );
l_long file_write_double ( p_file o, l_double i );
l_long file_write_char ( p_file o, l_char i );
l_long file_write_vformated ( p_file o, l_text str, VA_LIST arguments );
l_long file_write_formated ( p_file o, l_text str, ... );

/*
   Reads the string from the file that will previously stored
   by function file_write_string and was stored at the same position as the
   current position of reading string is.
   @ptr - text returned by the function ended by '\0'

   Returns the size of bytes succesfull read. Please note that this size
   is not the size of string in the case of F_BINARY,
   but by default the size of string+4+1.

   Note :
   The memory returned in @ptr by this call must be freed by the caller.
*/
l_long file_read_string ( p_file o, l_text *str );

l_long file_read_int ( p_file o, l_int *i );
l_long file_read_short ( p_file o, l_short *i );
l_long file_read_long ( p_file o, l_long *i );
l_long file_read_float ( p_file o, l_float *i );
l_long file_read_double ( p_file o, l_double *i );
l_long file_read_char ( p_file o, l_char *i );

/*
  Returns the file in the character array ended by zero.
  If file is not text-file be carefull, because it may contains 0
  anywhere in the file, so use @sz as length of the file, not string_length
*/
l_text file_to_string ( p_file o, l_long *sz );


/*
  This function can be used to indicate if the given FILE is at the
  end-of-file or not.

  Returns Nonzero at end-of-file, zero otherwise.
*/

l_int file_eof ( p_file o );


/*
  Get one character from FILE.
  The character ([0..255]) or `-1' if eof or error.
*/
l_int  file_getc ( p_file o );


/*
  The value read, or `-1' for end-of-file or error.  Since `-1' is a
  valid integer, you should use `file_eof' to detect this
  situation.
*/
l_int file_getw ( p_file o );

/*
  Tests if the file was succesfull opened
*/
l_bool file_is_open ( p_file o );


/**
 * Makes new file @o and sets information about the link to this file
 * and returns true on success.
*/
l_bool  file_set_link ( p_file o, t_filelink fl );


/**
 * Returns true if file @o is link.
*/
l_bool  file_is_link ( p_file o );


/**
 * Releases memory of @fl that was get by "file_get_link" function.
*/
void   file_free_link ( p_filelink fl );


/**
 * Gets information about link if the file @o is link to directory
 * or to file.
*/
p_filelink file_get_link ( p_file o );


/**
 * If the file is link then returns link file name otherwise normal path
*/
l_text file_get_direct ( l_text path );


/**
 * Makes new link to file @o and placed the link to directory path.
*/
l_bool  file_set_link_to ( p_file o, l_text path );


/*
  This function set protocol automatically by path.
  See xprotocol.h where you will find protocols_get function.
  This function choose protocol from the list of
  protocols that was previously made and can be increased or decreased
  in run-time.
*/
p_file file_new ( l_text path );

p_file file_new2 ( l_text path, p_protocol p );
p_file file_init ( p_file o, l_text path, p_protocol p );
p_file file_duplicate ( p_file o, l_text newpath );
p_file file_temp ( l_text prefix, l_text sufix );


p_file file_free ( p_file o );
void   file_free2 ( l_ptr o );
p_file file_open ( p_file o, l_int flags );
p_file file_close ( p_file o );


#ifdef __cplusplus
} /* end of poor C code */
#endif

#endif /* end of xfile.h */

