/*****************************************************************/
/*                                                               */
/*                                                               */
/*  File     :     XMOUSE.H                                      */
/*                                                               */
/*  Producer :     Interactive Studio                            */
/*                                                               */
/*        Programmers :  Michal Stencl                           */
/*                                                               */
/*                                                               */
/*  Copyright (c) 2000, Interactive Studio                       */
/*  All Rights Reserved                                          */
/*                                                               */
/*  Contacts :                                                   */
/*                                                               */
/*        Address :  Kozia 29, Bratislava 81103, Slovak Republic */
/*        E-mail  :  developer@InteractiveStudio.sk              */
/*        WEB     :  http://www.InteractiveStudio.sk/            */
/*                                                               */
/*                                                               */
/*  This file is part of MatrixOS                                */
/*                                                               */
/*****************************************************************/

/*
  First release : 06.09.2000 by Michal Stencl
*/

#ifndef __XMOUSE_H_INCLUDED__
#define __XMOUSE_H_INCLUDED__

#ifdef __cplusplus
extern "C" { /* begin poor C code */
#endif

#include"basics.h"
#include"drvbas.h"


/* mouse system cursors */
#ifndef CUR_ARROW
#define CUR_ARROW            0x00
#endif
#ifndef CUR_MOVE
#define CUR_MOVE             0x01
#endif
#ifndef CUR_GROW
#define CUR_GROW             0x02
#endif
#ifndef CUR_TEXT
#define CUR_TEXT             0x03
#endif
#ifndef CUR_STOP
#define CUR_STOP             0x04
#endif
#ifndef CUR_DRAG
#define CUR_DRAG             0x05
#endif
#ifndef CUR_CLOCK
#define CUR_CLOCK            0x06
#endif
#ifndef CUR_DRAGH
#define CUR_DRAGH            0x07
#endif
#ifndef CUR_PENCIL
#define CUR_PENCIL           0x08
#endif
#ifndef CUR_TARGET
#define CUR_TARGET           0x09
#endif
#ifndef CUR_TARGET2
#define CUR_TARGET2          0x0a
#endif
#ifndef CUR_ZOOM
#define CUR_ZOOM             0x0b
#endif
#ifndef CUR_SCRUP
#define CUR_SCRUP            0x0c
#endif
#ifndef CUR_SCRDN
#define CUR_SCRDN            0x0d
#endif
#ifndef CUR_SCRLF
#define CUR_SCRLF            0x0e
#endif
#ifndef CUR_SCRRG
#define CUR_SCRRG            0x0f
#endif
#ifndef CUR_TEXT2
#define CUR_TEXT2            0x10
#endif
#ifndef CUR_TEXT2
#define CUR_TEXT2            0x11
#endif
#ifndef CUR_FINGER
#define CUR_FINGER           0x12
#endif
#ifndef CUR_HAND
#define CUR_HAND             0x13
#endif

#define CURSOR_GETCURSOR(data,n)    (p_cursor)vector_element(data, n)
#define CURSOR_GETCURSORWIDTH(d,n)  low_grx_width((p_cursor)vector_at(data, n))->bitmap[0])
#define CURSOR_GETCURSORHEIGHT(d,n) low_grx_height((p_cursor)vector_at(data, n))->bitmap[0])
#define CURSOR_GETSYSTEM(n)         CURSOR_GETCURSOR(cursor_system,n)
#define CURSOR_GETSYSTEMWIDTH(n)    CURSOR_GETCURSORWIDTH(cursor_system, n)
#define CURSOR_GETSYSTEMHEIGHT(n)   CURSOR_GETCURSORHEIGHT(cursor_system,n)

#define MOUSE_ISDELAY(d)       false//(time_diff_mili(OBJECT(mouse)->process_time) > (d))

#define  REG_MOUSE     "mouse"

/* the differenc between two clicks that's still defined as doubleclick */
#define  MOUSE_TIME_DOUBLECLICK  500

/* cursor was showed */
#define  MO_SF_MOUSESHOW         0x0001000
/* cursor was hiden */
#define  MO_SF_MOUSEHIDE         0x0002000
/* left mouse button is still pressed */
#define  MO_SF_MOUSELAUTO        0x0004000
/* middle mouse button is still pressed */
#define  MO_SF_MOUSEMAUTO        0x0008000
/* right mouse button is still pressed */
#define  MO_SF_MOUSERAUTO        0x0010000
/* left mouse button was doubleclicked */
#define  MO_SF_MOUSELDOUBLE      0x0020000
/* right mouse button was doubleclicked */
#define  MO_SF_MOUSERDOUBLE      0x0040000
/* middle mouse button was doubleclicked */
#define  MO_SF_MOUSEMDOUBLE      0x0080000
/* any of mouse buttons was doubleclicked */
#define  MO_SF_MOUSEDOUBLE       (MO_SF_MOUSELDOUBLE|MO_SF_MOUSERDOUBLE|MO_SF_MOUSEMDOUBLE)
/* any of mouse buttons is still pressed */
#define  MO_SF_MOUSEAUTO         (MO_SF_MOUSELAUTO|MO_SF_MOUSERAUTO|MO_SF_MOUSEMAUTO)
/* any of mouse buttons was pressed */
#define  MO_SF_MOUSEDOWN         (MO_SF_MOUSELDOWN|MO_SF_MOUSERDOWN|MO_SF_MOUSEMDOWN)
/* any of mouse buttons was released */
#define  MO_SF_MOUSEUP           (MO_SF_MOUSELUP|MO_SF_MOUSERUP|MO_SF_MOUSEMUP)
/* any of mouse buttons is down */
#define  MO_SF_MOUSEPRESS        (MO_SF_MOUSEAUTO|MO_SF_MOUSEDOWN)
/* left mouse button is down */
#define  MO_SF_MOUSELPRESS       (MO_SF_MOUSELAUTO|MO_SF_MOUSELDOWN)
/* middle mouse button is down */
#define  MO_SF_MOUSEMPRESS       (MO_SF_MOUSEMAUTO|MO_SF_MOUSEMDOWN)
/* right mouse button is down */
#define  MO_SF_MOUSERPRESS       (MO_SF_MOUSERAUTO|MO_SF_MOUSERDOWN)

/*
  Object t_mouse is used for getting information
  from mouse, or set cursors, speed,
  range, etc... This object is placed in t_driver object
  together with t_keyboard. Whenever is OBJECT(driver)->translate_event...
  called it calls also translate_event function from t_mouse structures.
  If the user operate with the mouse, it sets t_object.state to mouse
  flags ...see above.
*/
/*
  Translates events from mouse. If mouse was moved or the button was
  pressed or released or is pushing, @type of event is set to
  EV_MOUSE, @t_event.obj is set to @o, position of mouse is set to
  t_mouse.where and important states are set in t_object.state.
  ( see MO_SF_XXXX ).
*/

typedef struct t_mouse *p_mouse;
typedef struct t_mouse {

  /* inherite functions / variables from t_object class */
  struct t_object  obclass;

  /* speed of mouse, speed.x = 0 is maximal speed in x coord and
     speed.x = 10 is minimal speed. It's same in speed.y
  */
  struct t_point   speed;

  /* point of mouse, where cursor is placed on the screen */
  struct t_point   where;

  /* number of buttons are available for mouse */
  l_char           buttons;

  l_dword          state_last;
  l_int            dclick_milisec;
  l_big            dclick_old_time;


  /*
    Returns current cursor of the mouse.
  */
  p_cursor  (*get_cursor)(p_mouse o);

  /*
    Returns the range were mouse is visible
  */
  t_rect    (*get_range)(p_mouse o);

  /*
    Returns the focus relative to mouse cursor.
  */
  t_point   (*get_focus)(p_mouse o);

  /*
    Tests if mouse is visible
  */
  l_bool    (*is_visible)(p_mouse o);

  /*
    Shows mouse cursor ( makes it visible ) to the screen.
    Returns true if mouse was succesfull shown, otherwise false.
  */
  l_bool    (*show)(p_mouse o);

  /*
    Hides mouse cursor ( makes it unvisible ) from the screen.
    Returns true if mouse was succesfull hiden, otherwise false.
  */
  l_bool    (*hide)(p_mouse o);

  /*
    Sets the mouse mode @mode in color @color and rect @r.

    You can choose from the following modes :

    MO_MO_RECT - efect like drag-window frame
    MO_MO_GROW - efect like resize-window frame

    Returns true if succesfull
  */
  l_bool    (*set_mode)(p_mouse o, l_int mode, l_color col, t_rect r );

  l_bool    (*set_mode_ex)(p_mouse o, void (*mode_draw_func)(p_bitmap bmp, l_ptr ptr, l_rect x1, l_rect y1, l_rect x2, l_rect y2, l_color color ), l_ptr ptr, l_color col, t_rect r );

  /*
    Sets the range where mouse is visible
  */
  void      (*set_range)(p_mouse o, t_rect r );

  /*
    Sets the time range where doubleclick is succesfull
  */
  void      (*set_dclick_diff)(p_mouse o, l_int mili);

  /*
    Sets the cursor of the mouse and returns true if succesfull.
  */
  l_bool    (*set_cursor)(p_mouse o, p_cursor cursor);

  /*
    Sets the speed of x and y coords.
  */
  void      (*set_speed)(p_mouse o, t_point speed );

  /*
    Sets mouse' cursor to new point. At this point has mouse focus.
  */
  void      (*set_pos)( p_mouse o, t_point where );

  /*
    Blocks mouse in range @r and returns value used for unblocking.
    Usage :
    If the mouse cursor is visible and it's placed in this area, mouse
    will be hiden, otherwise without efect. Used in graphics function, when
    something is drawn to the screen and we don't want to hide and show mouse
    each time. We only blocks the range we are going to draw and in the case
    the mouse is placed in this range, it will be hiden.
  */
  l_int     (*block)(p_mouse o, t_rect r );

  /*
    Unblocks mouse that was previously blocked by function t_mouse.block.
    @i - the value returned from function t_mouse.block
  */
  void      (*unblock)(p_mouse o, l_int i );

  /*
    Tests if mouse is blocked and hiden.
  */
  l_bool    (*is_block)(p_mouse o );

  /*
    Sets the focus of the mouse relative to the cursor in the case the
    focus is not the same as old one. This function need to hide mouse
    then sets the focus and shows it back to the screen, of-course if it
    isn't hide.
  */
  void      (*set_focus)(p_mouse o, t_point focus );

} t_mouse;

extern p_mouse  mouse;
extern p_mouse  def_mouse;

#define MOUSE(o) ((p_mouse)(o))
#define DEF_MOUSE_FUNC(y) DEF_FUNC(def_mouse,y)

/* system cursors */
extern p_vector   cursor_system;
/* default cursor */
extern p_cursor   cursor_standard;

/*
  Sets cursor from standard cursor' types, where (id) may be one of
  CUR_XXXX values, see above.
*/
#define  mouse_set_cursor_id(id)        mouse->set_cursor(mouse, CURSOR_GETSYSTEM(id))
#define  mouse_rect_block(ctx,i,r)      mouse_block##i = mouse->block(mouse, ctx, r)
#define  mouse_rect_unblock(i)          mouse->unblock(mouse, i)

/*
  Installs mouse
*/
extern p_mouse  (*mouse_init) ( p_mouse o );
p_mouse  _mouse_init ( p_mouse o );


/*
  Sets system cursors from the resource file @cursorfile and returns true
  if succesfull
*/
l_bool    mouse_set_system_cursors ( l_text cursorfile );


/*
  Loads system cursors from the file what information about this file
  is stored in system registers.

  Returns true if system cursors was succesfull loaded.
*/
l_bool    mouse_load_cursors ( void );


#ifdef __cplusplus
} /* end of poor C code */
#endif

#endif /* end of xmouse.h */

