/*****************************************************************/
/*                                                               */
/*                                                               */
/*  File     :     XOBJECT.H                                     */
/*                                                               */
/*  Producer :     Interactive Studio                            */
/*                                                               */
/*        Programmers :  Michal Stencl                           */
/*                                                               */
/*                                                               */
/*  Copyright (c) 2000, Interactive Studio                       */
/*  All Rights Reserved                                          */
/*                                                               */
/*  Contacts :                                                   */
/*                                                               */
/*        Address :  Kozia 29, Bratislava 81103, Slovak Republic */
/*        E-mail  :  developer@InteractiveStudio.sk              */
/*        WEB     :  http://www.InteractiveStudio.sk/            */
/*                                                               */
/*                                                               */
/*  This file is part of MatrixOS                                */
/*                                                               */
/*****************************************************************/

/*
  First release : 06.09.2000 by Michal Stencl
*/

#ifndef __XOBJECT_H_INCLUDED__
#define __XOBJECT_H_INCLUDED__

#ifdef __cplusplus
extern "C" { /* begin poor C code */
#endif

#include"basics.h"
#include"drvbas.h"


/* phase settings */

#define PH_PREPROCESS         0x0000001
#define PH_POSTPROCESS        0x0000002
/* If object has process but the t_object had process before. */
#define PH_INACTIVEPROCESS    0x0000004


/* options settings */

/* object is over the all objects */
#define OB_OF_TOPSELECT       0x0000001
/* object is enable for group */
#define OB_OF_ENABLE          0x0000002
/* object can be selected */
#define OB_OF_SELECTABLE      0x0000004
#define OB_OF_NOTACTIVATE     0x0000008
/* object is still in process */
#define OB_OF_STILLPROCESS    0x0000010
/* object is still in process */
#define OB_OF_CANTDELETED     0x0000020
/* */
#define OB_OF_THREADSTORE     0x0000040
/* */
#define OB_OF_WAITFOREND      0x0000080
/* This flags can select this object without
   deselecting other one. Used in menus. */
#define OB_OF_SELECTDES       0x0000100


/**
 * Data flags
*/
/* Can write data */
#define DATAF_SET             0x0000001
/* Can read data */
#define DATAF_GET             0x0000002
/* Can write data as link */
#define DATAF_SETLINK         0x0000004


/* tag settings */

#define TAG_WAITFOREND        0x0080000
#define TAG_DISPOSE           0x0010000
#define TAG_NONE              0x0008000
#define TAG_OBJECT            0x0000001
#define TAG_VIEW              0x0000002
#define TAG_WINDOW            0x0000004
#define TAG_LISTCHILD         0x0000008
#define TAG_BUTTON            0x0000010
#define TAG_ISVALID(t)        ((t)<TAG_DISPOSE)


/* state settings */

/* process is call only for this object */
#define  OB_SF_MODAL          0x0000001
/* object is visible */
#define  OB_SF_VISIBLE        0x0000002
/* object is select */
#define  OB_SF_SELECTED       0x0000004
/* object is focused */
#define  OB_SF_FOCUSED        0x0000008


/*
  event types
*/
#define  EV_NOTHING           0x0000
#define  EV_MOUSE             0x0001
#define  EV_KEYBOARD          0x0002
#define  EV_MESSAGE           0x0004
#define  EV_INFO              0x0008
#define  EV_RETYPE            0x0010
#define  EV_VIEWS             0x0020
#define  EV_BROADCAST         0x0040

#define  EV_IGNORE_ACTIVE_PROCESS(x) false


/* standard retype messages */

/* use with EV_RETYPE */

/* this message redraw files, where event->info = path to file */
#define  MSG_RETYPEPATH       1

#define  RETYPE_FILES_IN_PATH(p,free_info)  \
     obj_post_message_put(EV_RETYPE, MSG_RETYPEPATH, NULL, (p), free_info)


/* standard views messages */

/* use with EV_VIEWS */
/* this signals to owner, that bounds of sub-view was changed */
#define  MSG_CHANGEDBOUNDS    1


/*
  standard messages
*/
#define  MSG_NOTHING          0
#define  MSG_QUIT             1
#define  MSG_CLOSE            2
#define  MSG_CANCEL           3
#define  MSG_COPY             4
#define  MSG_PASTE            5
#define  MSG_DEL              6
#define  MSG_OK               7
#define  MSG_YES              8
#define  MSG_NO               9
#define  MSG_HELP            10

#define  MSG_HISTORY         11
#define  MSG_SELECTALL       12

#define  MSG_CONTINUE        13
#define  MSG_EXIT            14
#define  MSG_YESTOALL        15
#define  MSG_NOTOALL         16

#define  MSG_SCROLLUP        17
#define  MSG_SCROLLDOWN      18
#define  MSG_SCROLLLEFT      19
#define  MSG_SCROLLRIGHT     20

#define  MSG_CUT             21
#define  MSG_UNDO            22
#define  MSG_REDO            23

#define  MSG_DONE            24

/* If the view not performed the mouse events and wants to send it to
   view that's may be placed under this view, it may use this message to
   continue */
#define  MSG_MOUSECONTINUE   25

#define  MSG_STOP            26
#define  MSG_DISPOSE         27

/*
  data style
*/

/* get all data from object */
#define  DS_ALL        0x00000001
/* get selected data from object */
#define  DS_SELECTED   0x00000002
/* get data that object want to send */
#define  DS_WHATEVER   0x00000004
/* delete data from object call for set_data */
#define  DS_DELETE     0x00000008
/* cuts data from the source to destination */
#define  DS_CUT        0x00000010
/* copy data from the source to destination */
#define  DS_PASTE      0x00000020



/*
  data structure
*/
/* main program, must be defined in program.c file */

extern  struct t_object   program;
extern  p_vector program_delete_array;
extern  p_vector program_close_array;

typedef struct t_object *p_object;

typedef struct t_data *p_data;
typedef struct t_data {

  /* definition of data - see filter.h for basics definitinos DAT_XXXX */
  l_tag     id;
  /* style of transporting : DS_XXXX - used only in getting */
  l_dword   style;
  /* data of transport */
  l_ptr     ptr;
  /* object that transported data */
  p_object  info_obj;

  /* This function keeps the information how free data of @ptr. It's
     important to set this function for the same function that can release
     data @ptr. It's used by caller. */
  void     (*free) ( l_ptr ptr );

  /* Internal flags */
  l_dword   internal1;

  l_char    reserved[12];

} t_data;


/* clipboard */

extern p_vector  clipboard;
extern l_int     clipboard_max;

/*
  Inserts data @src into clipboard

  Returns ZERO on success
*/
l_int    clipboard_insert_as ( p_data src );


/*
  Inserts pointer @ptr of size @size of type @id into clipboard.

  Available @id :

  DAT_TEXT      "TEXT"     single text ended by zero
  DAT_IMAGE     "IMAGE"    type of BITMAP
  DAT_TFILE     "TFILE"    file t_file structure
  DAT_ALLKNOW   "ALLKNOW"  all data types are accepted
  DAT_NUMBER    "NUMBER"   type is the number
  DAT_FLOAT     "FLOAT"    type is the number with floating point
  DAT_HTML      "HTML"     type is the HTML format
  DAT_VECTOR    "VECTOR"   type is vector
  .....etc...

  Returns ZERO on success.
*/
l_int    clipboard_insert ( l_tag id, l_ptr data, l_long size );


/*
  Returns data inserted into clipboard at index @pos
*/
p_data   clipboard_get_at ( l_int pos );


/**
 * Returns true if the last item in the clipboard is known by
 * object @o in the thing of @dflags. See DATAF_XXXX for more information.
 *
 * ( Returns true if object @o understand clipboard's data type )
*/
l_bool   clipboard_same ( p_object o, l_int dflags );


/*
  Returns the last inserted data.
*/
p_data   clipboard_get ( void );


/*
  event structure
*/
typedef struct t_event *p_event;
typedef struct t_event {

  /* type of event, see EV_XXXX */
  l_dword      type;
  /* message of event, use for your own application events from 10001.. */
  l_dword      message;
  /* pointer to object that made event */
  p_object     obj;
  /* time when the event was inserted into */
  l_timestamp  time;
  /* info of event */
  l_ptr        info;
  /* Function tha can free info @info */
  void         (*free_info) ( l_ptr info );

  l_char       reserved[20];

} t_event;


extern t_event   event_main;

/*--defint*/
extern l_big     event_timer;
extern p_object  event_stop;
/*--notint*/

/* process definition */

extern l_bool go_process;

/* !!! warning !!! can't be NULL */
extern l_int (*program_int) ( l_int i );


extern void (*program_delete_old) ( void );

/*
  main object structure
*/
typedef struct t_object {

  /* tag of object = id of object type */
  l_dword   tag;
  /* current state of object, see  XX_SF_XXXXX */
  l_dword   state;
  /* options of object, see XX_OF_XXXXX */
  l_dword   options;
  /* the message returned at the end of process - see "valid", "execute" */
  l_dword   end_state;
  /* current phase of process for child objects */
  l_int     phase;
  l_int     tick;

  /* */
  l_process   process;
  l_long      waitings;
  l_timestamp process_time;
  l_int       process_tick;

  /* the parent of object */
  p_object  owner;
  /* next object */
  p_object  next;
  /* previous object */
  p_object  prev;
  /* the last child object */
  p_object  last;
  /* the child object with the highest priority */
  p_object  prefer;

  l_char    reserved[48];

/*
  Uninstall object @o and removes it from group. If the options was set
  to OB_OF_WAITFOREND, it isn't installed while o->options is set
  to OB_OF_WAITFOREND. If object returns false and o->tag is set to
  TAG_WAITFOREND, we cannot free the memory of @o while o->options is
  set to OB_OF_WAITFOREND.

  Returns true on success, otherwise it returns false
*/
  l_bool     (*done) ( p_object o );


/*
  Returns the first occurence of child view from child @o that fulfill
  options settings @opt, state settings @sta and is the first in left
  from @o if @forward is false or first in the right if @forward is true.

  Ignores testing of @sta or @opt if ZERO.

  If no view was found, returns NULL

  Note, @o can be also object not view inherited class.
*/
  p_object   (*find_match_view) ( p_object o, l_dword sta, l_dword opt, l_bool forward );


/*
  Returns the first occurence of child object from child @o that fulfill
  options settings @opt, state settings @sta and is the first in left
  from @o if @forward is false or first in the right if @forward is true.

  Ignores testing of @sta or @opt if ZERO.

  If no object was found, returns NULL
*/
  p_object   (*find_match) ( p_object o, l_dword sta, l_dword opt, l_bool forward );


/*
  Returns the first parent of @o that's inherited from class t_view. If
  no view is above this object, returns NULL.

  NOTE:
  For object inherited classes, use o->owner.
*/
  p_object   (*owner_view) ( p_object o );


/*
  Returns the next inherited view from the object/view @o.
  If the view wasn't
  found and @o isn't view too, returns NULL. This continue while not found
  the view. If all objects was tests and no view was found, then
  return NULL.

  NOTE:
  For object inherited classes, use o->next.
*/
  p_object   (*next_view) ( p_object o );


/*
  Returns the previous inherited view from the object/view @o.
  If the view wasn't found and @o isn't view too, returns NULL.
  This continue while not found the view.
  If all objects was tests and no view was found, then return NULL.

  NOTE:
  For object inherited classes, use o->prev.
*/
  p_object   (*prev_view) ( p_object o );


/*
  Returns the last inherited child view in the parent object/view @o.
  If the view wasn't found and @o isn't view too, returns NULL.
  This continue while not found the view.
  If all childs was tests and no view was found, then return NULL.
  It checks also childs of childs, etc...

  NOTE:
  For object inherited classes, use o->last.
*/
  p_object   (*last_view) ( p_object o );


/*
  Returns the first inherited child view in the parent object/view @o.
  If the view wasn't found and @o isn't view too, returns NULL.
  This continue while not found the view.
  If all childs was tests and no view was found, then return NULL.
  It checks also childs of childs, etc...

  NOTE:
  For object inherited classes, use o->first(o).
*/
  p_object   (*first_view) ( p_object o );


/*
  The same as o->next, but if the @o is the last view,
  then returns NULL
*/
  p_object   (*next_to_last) ( p_object o );


/*
  The same as o->prev, but if the @o is the first view,
  then returns NULL
*/
  p_object   (*prev_to_first) ( p_object o );


/*
  The same as prev_view, but if the @o is the first view,
  then returns NULL
*/
  p_object   (*prev_view_to_first) ( p_object o );


/*
  The same as next_view, but if the @o is the last view,
  then returns NULL
*/
  p_object   (*next_view_to_last) ( p_object o );

  /**
   * Sets object ability.
  */
  void       (*set_enable) ( p_object o, l_bool enable );

/*
  Returns the top object ( grand parent of all parents and their childs ).
  Mainly it's @program.
*/
  p_object   (*top) ( p_object o );

/**
  * Return true if object @o is connected with other objects.
*/
  l_bool     (*connected) ( p_object o );

/*
  Setups the object @o and if it has an parent ( only @program object
  has no parent, but all of yours must have ) , set preferences of the
  parent to this object if was inserted as the first or makes nothing.

  Use this function in all inherited objects that wants to set some
  values or call some functions after inserting into parent. For example
  t_view use this function for showing the view.
*/
  void       (*setup) ( p_object o );


  void       (*after_init) ( p_object o );


/*
  Inserts the object @sub as the child into the parent @o.
  By default, inserts object @sub as the last with the lowest priorite of
  process. If @sub->options is set to OB_OF_TOPSELECT, it's moved to
  the highest priority. If @sub->options is set to OB_OF_STILLPROCESS,
  it calls function @sub->init_still_process ( see above ).

  Returns @sub.
*/
  p_object   (*insert) ( p_object o, p_object sub );


/*
  The low-level function must be called from insert function.

  Inserts object @sub as the child of object @o before the object @before.
  If @before is NULL, @sub is inserted as the last object in parent' group.

  This function is the first, where the @sub parent is set to @o. It makes
  the conectivity with the others children in parent' group.

  Returns @sub.
*/
  p_object   (*insert_before) ( p_object o, p_object sub, p_object before );


/*
  Moves the object @o in front of object @before and makes the best
  preferences in parent' group.
*/
  void       (*put_in_front_of) ( p_object o, p_object before );


/*
  Removes the child object @sub from the parent @o. Not release the
  memory, it's the opposite function to "insert". It only destroies
  conectivity with the parent and other parent' child objects ( brothers ).
  If @sub was the first object it makes the first as @sub->next. If it
  was the last one, it makes the last as @sub->prev.
*/
  void       (*remove) ( p_object o, p_object sub );


/**
 * This function returns TRUE if object can do something with @data_type
 * according to @dflags as DATA flags.
 *
 * dflags may be combination of followings :
 *
 * DATAF_SET    - object can read data of @data_type by using function
 *                "get_data"
 * DATAF_GET    - object can write data of @data_type by using function
 *                "set_data"
 *
 */
  l_bool     (*know_data) ( p_object o, l_tag data_type, l_dword dflags );
/**
 * Copies data from the object @o to the clipboard and returns
 * TRUE on success.
*/
  l_bool     (*copy_data) ( p_object o );
  l_bool     (*cut_data) ( p_object o );
  l_bool     (*paste_data) ( p_object o );
  l_bool     (*delete_data) ( p_object o );

  l_bool     (*thread_init) ( p_object o );
  l_bool     (*thread_done) ( p_object o );
  l_bool     (*thread_wait_init) ( p_object o );
  l_bool     (*thread_wait_done) ( p_object o );
  l_bool     (*has_thread) ( p_object o );


/*
  Returns the first child object in @o group. The first object is the same
  as o->last->next if the last ( any ) exists.
*/
  p_object   (*first) ( p_object o );


/*
  Selects the object if the o->option are set to
  OB_OF_ENABLE+OB_OF_SELECTABLE and o->state are set to OB_SF_VISIBLE.

  If options are set also to OB_OF_TOPSELECT, put this object in front of
  all others what means it makes it first in parent' group and
  puts only the best preference. If options are not set to OB_OF_TOPSELECT,
  only puts the best preference to this object in parent' group.

  If the object was succesful selected, returns true.
*/
  l_bool     (*select) ( p_object o );


/*
  Sets states o->state to @st if @set is true, otherwise it unsets @st.
  If @st is OB_SF_FOCUSED, set/unset the same focus to first child with
  state OB_SF_SELECTED+OB_SF_FOCUSED and to its childs too, etc...

  For example : If the windows is selected, call the first child in the
  windows that was selected at the last time and makes the focus to it.
*/
  void       (*set_state) ( p_object o, l_dword st, l_bool set );


/*
  Sets options o->options to @op if @set is true, otherwise it unsets @op.
*/
  void       (*set_options) ( p_object o, l_dword op, l_bool set );


/*
  Tests if object @o includes states @st in o->state
*/
  l_bool     (*is_state) ( p_object o, l_dword st );


/*
  Tests if object @o includes options @op in o->options
*/
  l_bool     (*is_options) ( p_object o, l_dword op );


/*
  Returns child object that is placed at @index in the parent @o. If
  @index is negative number or is higher than number of childs-1, returns
  NULL.
*/
  p_object   (*at) ( p_object o, l_long index );


/*
  Returns index of child object @p in the parent @o. If the child or
  parent not exists or child wasn't found in the parent, returns -1.
*/
  l_long     (*index_of) ( p_object o, p_object p );

/**
 * If this function calls some drawing primitives, or function that calls
 * some, it must test if the view is visible by function
 * VIEW(s)->is_visible(VIEW(s)).
*/
  void       (*func_callback) ( p_object s );


/*
  Puts the object @s into still procecess objects what makes it sure,
  it will get the process each s->process_tick milisecond.
  @o must be the same
  object as @s - it's important for internal process of this function,
  because it calls all parent's put_into_stillprocess with @o
  until found the top object that control processes.

  Returns ZERO if success.
*/
  l_int      (*put_into_stillprocess) ( p_object o, p_object s );


/*
  Clears the object from still processes list. After this function will
  have the object the same priorites than the others.

  Returns ZERO on success.
*/
  l_int      (*clear_from_stillprocess) ( p_object o, p_object s );


/*
  This function is called when some data are draged from the object,
  or data are copied from this object into clipboard ( see t_data )

  This function must be inherited. This one makes nothing,
  only returns false.

  Returns true, if data was copied from the object, else return false

  Example how the function may look like :

  if ( rec ) {

     rec->info_obj = o;

     l_tag_cpy(rec->id, _your_id);

     switch ( rec->style ) {

       case DS_SELECTED : {

            rec->data = _your_selected_new_pointer;

       }; break;

       case DS_ALL : {

          rec->data = _your_alldata_new_pointer;

          return true;

       }; break;

       case DS_WHATEVER : {

          rec->style = DS_SELECTED;

          return OBJECT(o)->get_data(OBJECT(o), rec);

       }; break;
     };

     rec->id = DAT_NONE;
  };
  return false;
*/
  l_bool     (*get_data) ( p_object o, t_data *rec );


/*
  This function is called when some data are droped into the object,
  or data are pasted into object from clipboard, etc. set_data function
  may set data only if o->id equals to (id) that you select in
  t_object.data_type variable.

  This function must be inherited. This one makes nothing,
  only returns false.

  @rec - is the pointer to t_data structure, that contains
         following information:

         l_tag   id;  - identification of data, may contains more
                        than one bit settings

         DAT_NONE     - none data type, it not contains any data
         DAT_TEXT     - t_data->data is text end by zero ( C text )
         DAT_IMAGE    - t_data->data contains pointer to Allegro BITMAP
         DAT_LIST     - t_data->data is pointer to p_list structure. See id
                        of this ( t_list ) structure.
         DAT_ALLKNOW  - each data @id is accepted, it's used in trash
                        BIN f.e.


         l_dword   style;

                      - styles of t_data paste/copy

         DS_ALL       - get all data from object in current (id).
         DS_SELECTED  - get only selected data from object
         DS_WHATEVER  - object select, what style it want to copy to
                        clipboard
         DS_DELETE    - delete data from object. ( This may be combined
                        with previous styles. DS_ALL|DS_DELETE => delete
                        all data from object, DS_SELECTED|DS_DELETE => delete
                        only what's selected in object...)

        l_ptr     data;

                      - pointer to data ( l_ptr = void ), you get what
                        type of pointer it's from (id).

        p_object  info_obj;

                      - pointer to object, from where were data copied
                        to clipboard.

  Returns true, if data was pasted into this object, else returns false

  Example how the function may look like :


  l_bool ok  = false;

  if ( rec && o->know_data(rec->id, DATAF_WRITE) ) {

     if ( rec->style & DS_DELETE ) {

       Deletes only if focused
       if ( o->is_state(o, OB_SF_FOCUSED) )  {

         if ( rec->style & DS_ALL ) {

               delete all

               ok = true;

         } else {

               Deletes selected data only

               if ( was_some_selected ) ok = true;
               else return false;
         };
       };

     } else {

        Data are pasted into the object

        paste_data;

        ok = true;
     };

     if ( ok ) {

         if_you_need_redraw_something;

         Some redraw functions;
     };
  };
  return ok;
*/
  l_bool     (*set_data) ( p_object o, t_data *rec );


/*
  Function is called if need to select data in the object.
  This function must be inherited. This one makes nothing,
  only returns false.

  @data_syle -  can be one of followings :

                DS_ALL - select all data in the object
                ...

  @set       - true if need to be selected, false if need to be unselected

  Returns true on success
*/
  l_bool     (*select_data) ( p_object o, l_int data_style, l_bool set );


/*
  Sets the highest priority for the object @prefer in the parent' group
  @o. If @so is @true, this unsets the old one, that had the highest
  priority before. This function also sets states of @prefer->state
  to OB_SF_SELECTED+OB_SF_FOCUSED in setting and unsets states
  OB_SF_SELECTED+OB_SF_FOCUSED of @old->states in unsetting.

  Whenever your inherited object gets the state OB_SF_FOCUSED to your
  set_state function you may be sure there was set or unset the highest
  priority ( see t_object.set_state )
*/
  void       (*set_prefer) ( p_object o, p_object prefer, l_int so );


/*
  Finds the object in the parent' group @o, that's the best for setting
  as higher priority object. By default it's the same as
  o->set_prefer(o->first(o), 1).
*/
  void       (*reset_prefer) ( p_object o );


/*
  Tests the validity of the object mainly at the end. It's used
  e.g. in application when @msg is close and it's going to close the
  application. This function call each application' child valid function
  with the same message and in the case, there is some not saved,
  this is the last chance when do so. If the child don't want to save
  it ( maybe cancel ), it returns the object of this child and stops the
  closing, otherwise returns NULL.
*/
  p_object   (*valid) ( p_object o, l_dword msg );


/*
  Returns the last event in @event occured from the
  event manager selected for this object @o,
  basicly it's t_drivers class.
*/
  void       (*get_event) ( p_object o, t_event *event );


/*
  Puts the event @event to the events manager, that can be
  later read by function get_event. If there wasn't put other event, before
  calling get_event and put_event, get_event cath this, else get_event
  returns event inserted before.
*/
  void       (*put_event) ( p_object o, t_event *event );


/*
  Makes the loop of the process for this object @o. This function is used
  whenever window is modal or in the case of menu, when the application
  wait for the value returned by the modal object. This is the main process
  function of the @program too. At the first gets events then
  translates events of @o and loop while o->end_state == 0.
  If o->end_state not equals to ZERO and o->valid(o, o->end_state) is ZERO (
  the valdiation is correct ) end this loop and returns @o->end_state.

  Whenever the modal window is going to be closed, it sets @o->end_state to
  MSG_CLOSE and makes validation process.

  Returns @o->end_state.
*/
  l_dword    (*execute) ( p_object o );


/*
  Translates events @event in the child object @o if the object @o
  has no process at the time OR event->obj == @o ( this event was
  made just for this object ) and event->type is EV_BROADCAST ( only
  the short message ) OR the object can't be activated ( o->options
  is set to OB_OF_NOTACTIVATE ) OR event tells us, we must to ignore
  active process and set the event also to it.

  if EV_IGNORE_ACTIVE_PROCESS(event->type) or event->type & EV_RETYPE,
  then event tells us we must ingore the active process.
*/
  void       (*play_process) ( p_object o, t_event *event );


/*
  Calls each child object in the group' @o by the event @event,
  in the case the child object has no process at the time.
*/
  void       (*for_each_event) ( p_object o, t_event *event );


/*
  Translate events in all childs. Firstly is called
  function PLAY_PROCESS(o->prefer, event) by the object o->prefer -
  - the object with the highest priority if hasn't a process yet,
  then are called all childs from o->first(o) to o->last,
  by using function o->for_each_event.

  Then tests event->type getting from the childs or from @event if no child
  translates one and checks if the type is EV_MESSAGE and @event->message
  is MSG_PASTE, MSG_COPY, or MSG_DEL.

  If the message is MSG_PASTE, do :

  Calls function o->set_data and paste the last data inserted into
  clipboard at the last time.

  If the message is MSG_COPY, do :

  Calls function o->get_data by t_data->state = DS_WHATEVER
  and inserts the data to the clipboard.


  If the message is MSG_COPY, do :

  Calls function o->set_data by t_data->state = DS_DELETE|DS_WHATEVER,
  what should erase data from the object, see "set_data".

  If some of these messages was translated, clears event @event.

  NOTE :

  Must clear events whenever your function translates the message. Use the
  function "clear_event"
*/
  void       (*translate_event) ( p_object o, t_event *event );

} t_object;

extern p_object def_object;

#define OBJECT(o) ((p_object)(o))
#define DEF_OBJECT_FUNC(y) DEF_FUNC(def_object,y)


/* other object functions */

/*
  Tests if the object @e is active in process - has the process.
*/
l_bool     obj_is_active ( t_object *e );


/**
 * Cuts data from source object defined in @data to destination object.
 * This function is help function for DS_CUT in set_data function.
 * If @pasting is set to true, pasting is before the deleting.
*/
l_bool  obj_cut_data_from ( p_object dst, p_data data, l_bool pastefirst );


/*
  Tests if object @e is one of child objects in object @o. It tests also
  children of children, etc...
*/
l_bool     obj_is_my_object ( t_object *o, t_object *e );

/*
  Tests if the message @e belongs to one of child objects in object @o.
  It tests also children of children, etc...
*/
#define    obj_is_my_message(o,e) obj_is_my_object(o, e->obj)

/* object functions */

extern t_object *(*obj_init) ( t_object *o );
t_object *_obj_init ( t_object *o );

/*
  Firstly calls o->done and if returns true, releases the context
  of @o. Returns NULL on success, otherwise false if @o not exists or
  o->done returns false. ( see t_object.done )
*/
t_object*  obj_dispose ( t_object *o );

l_bool     obj_can_dispose ( l_ptr e );


/*
  Disposes all child objects in parent's group @o. Calls function
  "done" for each object in the parent's group and set the o->tag to
  TAG_DISPOSE. This will not free the memory of objects.
*/
void       obj_dispose_all ( t_object *o );

/*
  Gets if the object @o exists.
*/
#define    obj_exist(o)  (o ? OBJECT(o)->tag: TAG_NONE)

/**
 * Returns true if object was interrupted.
*/
#define    obj_interrupted(o) (((obj_exist(o) == TAG_NONE) || (obj_exist(o) == TAG_DISPOSE) || (obj_exist(o) == TAG_WAITFOREND)) ? true : false)

/*
  Makes sure the object @o gets process each @milis milisecond.
*/
void       obj_init_stillprocess ( p_object o, l_int milis );

/*
  Opposite to function obj_init_stillprocess
*/
#define    obj_done_stillprocess(o) obj_init_stillprocess(o, -1)


/*
  Translates events in all objects, where @type is the type of the event,
  @message is the message that's need to be translated, @obj is object
  that calls this function. @info is any pointer.
*/
void       obj_message_all_info ( l_dword type, l_dword message, p_object obj, void *info );

/*
  Translates events in the object @o, where @type is the type of the event,
  @message is the message that's need to be translated, @obj is object
  that calls this function. @info is any pointer.
*/
void       obj_message_info ( p_object o, l_dword type, l_dword message, p_object obj, void *info );
#define    obj_message(o,t,m,ob)   obj_message_info(o,t,m,ob,NULL)
#define    obj_message_all(t,m,ob) obj_message_all_info(t,m,ob,NULL)

/*
  Translates events in the object @o, where @type is the type of the event,
  @message is the message that's need to be translated, @obj is object
  that calls this function. @info is any pointer.

  This store message in the message queue and is take only from main cycle
*/
void  obj_post_message_put ( l_dword type, l_dword message, p_object obj, l_ptr info, void (*free_info)(l_ptr info) );


/**
 * Called before process we want to play
*/
l_bool obj_pre_play_process ( p_object o, t_event *event );


/**
 * post play process can be run only in the case that pre process returns
 * true.
*/
l_bool obj_post_play_process ( p_object o, t_event *event );


/**
 * Gets last message that comes from obj_post_message_info
*/
t_event obj_post_message_get ( void );


/**
 * Clears info of message that was put to "obj_post_message_put" function.
 * Info is cleared by function "free_info".
*/
void obj_post_message_clear ( p_event event );


/*
  Sets the @event to the following arguments : @type, @message, @obj, and
  gets the current running time into the @event->time. Other arguments
  sets to ZERO.
  You can later get the difference  between that time and the old one,
  by using function "low_timer_diff_mili".

  NOTE :
  For more information see t_event structure.
*/
void       event_set ( t_event *event, l_dword type, l_dword message, p_object obj );

#define clear_event(t)   clear_type(t,sizeof(t_event))
#define event_clear(t)   clear_type(t,sizeof(t_event))


/*
  Sets the @event to the following arguments : @type, @message, @obj,
  @info and gets the current running time into the @event->time.
  You can later get the difference  between that time and the old one,
  by using function "low_timer_diff_mili".

  NOTE :
  For more information see t_event structure.
*/
void  event_set_info ( t_event *event, l_dword type, l_dword message, p_object obj, l_ptr info, void (*free_info)(l_ptr info) );


#define thread_new() program.execute(&program);

/*
  Multitasking "delay" function.

  Please not use in fnuction where something is drawn.
*/
#define thread_delay(x)  low_timer_thread_delay(x)
#define thread2_delay(x) low_timer_thread2_delay(x)

#define thread_main(e)  program.translate_event(&program, e)

#define thread_call(__o, __x)    \
  if ( !IS_ACTIVE_PROCESS(o) ) { \
        ##__x;                   \
  };

/*
  Multitasking "for" directive.

  Please not use in fnuction where something is drawn.
*/
#define    thread_for(t1,t2,t3)\
                  for((##t1);(##t2)&&program_int(1);(##t3))

#define    thread2_for(t1,t2,t3)\
                  for((##t1);(##t2)&&program_int(2);(##t3))
/*
  Multitasking "while" directive.

  Please not use in fnuction where something is drawn.
*/
#define    thread_while(t)\
                  while((##t)&&program_int(1))

#define    thread2_while(t)\
                  while((##t)&&program_int(2))

/*
  Multitasking "do" directive.

  Please not use in fnuction where something is drawn.
*/
#define    thread_do   do
#define    thread2_do  do

/**
 * Odporucam v pripade, ze __x je nejaky while tuto podmienku ktora je
 * vo while dat aj pred thread_sub a thread_modal. Inake to robi nejake
 * blbosti akoby ten prvy while vobec netestoval.
*/
#define    thread_sub(__o,__x)\
  SUB_CODE {\
      if ( __o ) {\
         OBJECT(__o)->thread_init(OBJECT(__o));\
      };\
      SUB_CODE {\
               ##__x;\
      } END_CODE;\
      if ( __o ) {\
         OBJECT(__o)->thread_done(OBJECT(__o));\
      };\
  } END_CODE


#define    connect_modal(__o,__x) \
  SUB_CODE {\
      if ( __o ) {\
         OBJECT(__o)->thread_wait_init(OBJECT(__o));\
         DEBUGCODE_printf("\nThread Wait Init");\
      }\
      SUB_CODE {\
               ##__x;\
      } END_CODE;\
      if ( __o ) {\
         OBJECT(__o)->thread_wait_done(OBJECT(__o));\
         DEBUGCODE_printf("\nThread Wait Done");\
      };\
  } END_CODE

#define   thread_modal connect_modal

/*
  Multitasking interrupt.

  Please not use in fnuction where something is drawn.
*/
#define INTMAIN(e)      program.translate_event(&program, e)

/* Internals */
#define PROCESS_PASSIVE 0
#define PROCESS_ACTIVE  (!PROCESS_PASSIVE)

#define ACTIVE_PROCESS(o)    OBJECT(o)->process++
#define PASSIVE_PROCESS(o)   OBJECT(o)->process = _GEZ(OBJECT(o)->process-1)
#define REMOVE_PROCESS(o)    OBJECT(o)->process = PROCESS_PASSIVE
#define GET_PROCESS(o)       OBJECT(o)->process
#define RESET_PROCESS(o,pr)  OBJECT(o)->process = (pr)
#define IS_PASSIVE_PROCESS(o) (!OBJECT(o)->process)
#define PLAY_PROCESS(o,ev)   OBJECT(o)->play_process(o, ev)
#define IS_OKTOSUBPROCESS(o) 1

#define STOP_PROCESS()  go_process = false
#define START_PROCESS() go_process = true


#ifdef __cplusplus
} /* end of poor C code */
#endif

#endif /* end of xobject.h */

