/*****************************************************************/
/*                                                               */
/*                                                               */
/*  File     :     XREGS.H                                       */
/*                                                               */
/*  Producer :     Interactive Studio                            */
/*                                                               */
/*        Programmers :  Michal Stencl                           */
/*                                                               */
/*                                                               */
/*  Copyright (c) 2000, Interactive Studio                       */
/*  All Rights Reserved                                          */
/*                                                               */
/*  Contacts :                                                   */
/*                                                               */
/*        Address :  Kozia 29, Bratislava 81103, Slovak Republic */
/*        E-mail  :  developer@InteractiveStudio.sk              */
/*        WEB     :  http://www.InteractiveStudio.sk/            */
/*                                                               */
/*                                                               */
/*  This file is part of MatrixOS                                */
/*                                                               */
/*****************************************************************/

/*
  First release : 06.09.2000 by Michal Stencl
*/

#ifndef __XREGS_H_INCLUDED__
#define __XREGS_H_INCLUDED__

#ifdef __cplusplus
extern "C" { /* begin poor C code */
#endif

#include"basics.h"
#include"drvbas.h"

#define REG_TEXT     1
#define REG_NUM      3
#define REG_PASSWD  10

typedef  l_text   l_regdata;
typedef  l_text   l_regout;
typedef  l_text   l_reglin;
typedef  l_text   l_regsec;
typedef  l_text  *l_regval;

typedef struct t_regrgb {

  l_int r; /* red */
  l_int g; /* green */
  l_int b; /* blue */
  l_int x; /* reserved */

} t_regrgb;

#define   register_free_section(data) register_close(data)
#define   register_free_line(line) register_close(line)
#define   register_free_out(out) register_close(out)
void      register_free_values ( l_regval values );

/*
  System register
*/
extern l_regdata register_system;

/*
  Opens register file and returns pointer you will later use in
  register_XXXX functions.

  NOTE :
  The memory returned by this call must be freed by the caller by function
  "register_close"
*/
l_regdata register_open ( l_text filename );


/*
  Closes the register file that was opened by function register_open.
  It's the same like the function _free.
*/
void  register_close ( l_regdata data );


/*
  Updates registers from the file @filename.
  If the registers was changed inside the file, you need to
  update @data by this function. Returns true if @data was sucesfull updated,
  otherwise returns false.
*/
l_bool register_update ( l_regdata *data, l_text filename );

/**
 * Reads section @iname of register @data and try to executes all programs
 * placed inside this section. Returns ZERO on success.
 *
 * Function @process_func is pointer to function that handles loading, where
 * the @cur is current position of max @sz files and @filename is the name of
 * current loading file.
*/
l_int    register_execute ( l_regdata data, l_text iname, void (**process_func)(l_long cur, l_long sz, l_text filename ) );


/*
  Reads section @iname of register @data and returns section you can later
  use by functions register_sXXXXX

  NOTE :
  The memory returned by this call must be freed by the caller by function
  "register_free_section"
*/
l_regsec register_read_section ( l_regdata data, l_text iname );


/*
  Reads data from section @iname, line @ixname of register @data and
  returns value succesfull read or NULL in error.

  NOTE :
  The memory returned by this call must be freed by the caller by function
  "register_free_out"
*/
l_regout register_read ( l_regdata data, l_text iname, l_text ixname );


/*
  Reads number from section @iname, line @ixname of register @data and
  returns number succesfull read or 0 in error.
*/
l_long  register_read_int ( l_regdata data, l_text iname, l_text ixname );


/*
  Reads string from section @iname, line @ixname of register @data and
  returns string succesfull read or NULL in error.

  NOTE :
  The memory returned by this call must be freed by the caller by function
  "_free" or "low_mem_free" or "string_free"
*/
l_text register_read_str ( l_regdata data, l_text iname, l_text ixname );


/*
  Reads password string from section @iname, line @ixname of
  register @data and returns string succesfull read or NULL in error.
  Password is not visible in register file. It's coded and this function
  encodes the password and returns encoded text.

  NOTE :
  The memory returned by this call must be freed by the caller by function
  "_free" or "low_mem_free" or "string_free"
*/
l_text register_read_pswd ( l_regdata data, l_text iname, l_text ixname );


/*
  Reads RGB triplet from section @iname, line @ixname of register @data and
  returns t_regrgb succesfull read.
*/
t_regrgb register_read_rgb ( l_regdata data, l_text iname, l_text ixname );


/*
  Reads number of lines in the section @data that can get by function
  register_read_section. Returns more than ZERO if there are some lines,
  or 0 if they aren't.
*/
l_int     register_sread_linenum ( l_regsec data );


/*
  Reads @line from section @data and returns value at the right side of
  line and @def as the left side of line. You can later use these values
  in functions register_lread_function register_lread_values.

  NOTE :
  The memory returned by this call ( also in @def ) must be freed by the
  caller by function "register_free_line"
*/
l_reglin  register_sread_line ( l_regsec data, l_reglin *def, l_int line );


l_regval  register_lread_function ( l_reglin line, l_regout *func );
l_regval  register_lread_values ( l_reglin value );
void      register_free_values ( l_regval values );


/*
  Reads data from section @data, line @ixname.

  NOTE :
  The memory returned by this call must be freed by the caller by function
  "register_free_out"
*/
l_regout register_sread ( l_regsec data, l_text ixname );


/*
  Reads password string from section @data, line @ixname and returns
  string succesfull read or NULL in error.
  Password is not visible in register file. It's coded and this function
  encodes the password and returns encoded text.

  NOTE :
  The memory returned by this call must be freed by the caller by function
  "free_string"
*/
l_text register_sread_pswd ( l_regsec data, l_text ixname );


/*
  Reads number from section @data line @ixname and
  returns number succesfull read or 0 in error.
*/
l_long   register_sread_int ( l_regsec data, l_text ixname );


/*
  Reads string from section @data, line @ixname and
  returns string succesfull read or NULL in error.

  NOTE :
  The memory returned by this call must be freed by the caller by function
  "_free" or "low_mem_free" or "string_free"
*/
l_text   register_sread_str ( l_regsec data, l_text ixname );


/*
  Reads RGB triplet from section @data, line @ixname and
  returns t_regrgb succesfull read.
*/
t_regrgb register_sread_rgb ( l_regsec data, l_text ixname );


/*
 * Reads section @iname of register file @filename and try to executes
 * all programs placed inside this section. Returns ZERO on success.
 *
 * Function @process_func is pointer to function that handles loading, where
 * the @cur is current position of max @sz files and @filename is the name of
 * current loading file.
*/
l_int    register_fexecute ( l_text filename, l_text iname, void (**process_func)(l_long cur, l_long sz, l_text filename ) );


/*
  Reads section @iname of register file @filename and returns section
  you can later use by functions register_sXXXXX

  NOTE :
  The memory returned by this call must be freed by the caller by function
  "register_free_section"
*/
l_regsec register_fread_section ( l_text filename, l_text iname );


/*
  Reads data from section @iname, line @ixname of register file @filename
  and returns value succesfull read or NULL in error.

  NOTE :
  The memory returned by this call must be freed by the caller by function
  "register_free_out"
*/
l_regout register_fread ( l_text filename, l_text iname, l_text ixname );


/*
  Reads password string from section @iname, line @ixname of
  register file @filename and returns string succesfull
  read or NULL in error. Password is not visible in
  register file. It's coded and this function encodes the
  password and returns encoded text.

  NOTE :
  The memory returned by this call must be freed by the caller by function
  "_free" or "low_mem_free" or "string_free"
*/
l_text register_fread_pswd ( l_text filename, l_text iname, l_text ixname );


/*
  Reads number from section @iname, line @ixname of register file
  @filename and returns number succesfull read or 0 in error.
*/
l_long  register_fread_int ( l_text filename, l_text iname, l_text ixname );


/*
  Reads string from section @iname, line @ixname of register file
  @filename and returns string succesfull read or NULL in error.

  NOTE :
  The memory returned by this call must be freed by the caller by function
  "_free" or "low_mem_free" or "string_free"
*/
l_text register_fread_str ( l_text filename, l_text iname, l_text ixname );


/*
  Reads RGB triplet from section @iname, line @ixname of register file
  @filename and returns t_regrgb succesfull read.
*/
t_regrgb register_fread_rgb ( l_text filename, l_text iname, l_text ixname );


/*
  Writes data @value of type @type to section @iname, line @ixname of
  register file @filename.

  Type can be one of the followings :

  REG_TEXT     1
  REG_HEXA     2
  REG_DECA     3
  REG_PASSWD  10

  If the type is password it's important to use key, that will be used for
  encoding.

  Returns 0 on success.
*/
l_int  register_fwrite ( l_text filename, l_text inae, l_text ixname, l_ptr value, l_int type, l_text key );


/*
  Writes password @value to section @iname, line @ixname of
  register file @filename.

  It's important to use key, that will be used for encoding.

  Returns 0 on success.
*/
l_int  register_fwrite_pswd ( l_text filename, l_text iname, l_text ixname, l_text value, l_text key );


/*
  Writes string @value to section @iname, line @ixname of
  register file @filename.

  Returns 0 on success.
*/
l_int register_fwrite_str ( l_text filename, l_text iname, l_text ixname, l_text value );


/*
  Writes decadecimal number @value to section @iname, line @ixname of
  register file @filename.

  Returns 0 on success.
*/
l_int  register_fwrite_int ( l_text filename, l_text iname, l_text ixname, l_long value );

/*
  Writes hexadecimal number @value to section @iname, line @ixname of
  register file @filename.

  Returns 0 on success.
*/
l_int  register_fwrite_hex ( l_text filename, l_text iname, l_text ixname, l_long value );

/*
  Writes rgb triplet @rgb to section @iname, line @ixname of
  register file @filename.

  Returns 0 on success.
*/
l_int  register_fwrite_rgb ( l_text filename, l_text iname, l_text ixname, t_regrgb rgb );


#ifdef __cplusplus
} /* end of poor C code */
#endif


#endif /* end of xregs.h */

