/**
 * theader.h
*/

#ifndef __theader_h_included__
#define __theader_h_included__

#include<matrixos.h>

#include "in.h"

#ifdef __cplusplus
extern "C" { /* begin poor C code */
#endif

#define pack __attribute__ ((packed))

#define ETH_MSS 1400  /* MSS for Ethernet */


#define PD_ETHER 1
#define PD_SLIP  6

typedef l_byte l_eth_address[6];

#define MOV_MEM(s,d,c) low_mem_memmove(d,s,c)


#define SOCKESTABLISHED 1
#define SOCKDATAREADY   2
#define SOCKCLOSED      4


#define IP_TYPE     0x0008

/**
 * internal limitations
*/
#define MAX_GATE_DATA       12
#define MAX_NAMESERVERS     10
#define MAX_DOMAIN_LENGTH   80
#define MAX_COOKIES         10
#define MAX_STRING          50   /* most strings are limited */


#define MAXVJSA           1440 /* 10 s */
#define MAXVJSD            360 /* 10 s */
#define SAFETYTCP  0x538f25a3L
#define SAFETYUDP  0x3e45e154L

#define EL_INUSE        0x0001
#define EL_DELAY        0x0002
#define EL_TCP          0x0004
#define EL_SERVER       0x0008
#define EL_ASCII        0x0010
#define EL_NEVER        0x0020

/**
 * These are Ethernet protocol numbers but I use them for other things too
*/
#define UDP_PROTO  0x11
#define TCP_PROTO  0x06
#define ICMP_PROTO 0x01

#define TCP_MODE_BINARY  0       /* default mode */
#define TCP_MODE_ASCII   1
#define UDP_MODE_CHK     0       /* default to having checksums */
#define UDP_MODE_NOCHK   2       /* turn off checksums */
#define TCP_MODE_NAGLE   0       /* Nagle algorithm */
#define TCP_MODE_NONAGLE 4


/* The Ethernet header */
typedef struct {
    l_eth_address   destination pack;
    l_eth_address   source pack;
    l_ushort        type pack;
} eth_Header;


/* The Internet Header: */
typedef struct {
    unsigned        hdrlen  : 4 pack;
    unsigned        ver     : 4 pack;
    l_byte          tos pack;
    l_ushort        length pack;
    l_ushort        identification pack;
    l_ushort        frags;
    l_byte          ttl pack;
    l_byte          proto pack;
    l_ushort        checksum pack;
    l_dword         source pack;
    l_dword         destination pack;
} in_Header;

#define in_GetVersion(ip) ( (ip)->ver )
#define in_GetHdrlen(ip)  ( (ip)->hdrlen )  /* 32 bit word size */
#define in_GetHdrlenBytes(ip)  ( in_GetHdrlen(ip) << 2 ) /* 8 bit byte size */
#define in_GetTos(ip)      ( (ip)->tos)
#define in_GetTTL(ip)      ((ip)->ttl)
#define in_GetProtocol(ip) ((ip)->proto )

typedef struct {
    l_ushort    srcPort pack;
    l_ushort    dstPort pack;
    l_ushort    length pack;
    l_ushort    checksum pack;
} udp_Header;

#define UDP_LENGTH ( sizeof( udp_Header ))

typedef struct {
    l_ushort        srcPort pack;
    l_ushort        dstPort pack;
    l_dword         seqnum pack;
    l_dword         acknum pack;
    l_ushort        flags pack;
    l_ushort        window pack;
    l_ushort        checksum pack;
    l_ushort        urgentPointer pack;
} tcp_Header;

#define tcp_FlagFIN     0x0001
#define tcp_FlagSYN     0x0002
#define tcp_FlagRST     0x0004
#define tcp_FlagPUSH    0x0008
#define tcp_FlagACK     0x0010
#define tcp_FlagURG     0x0020
#define tcp_FlagDO      0xF000
#define tcp_GetDataOffset(tp) (tcpip_intel16((tp)->flags) >> 12)

/* The TCP/UDP Pseudo Header */
typedef struct {
    l_dword     src pack;
    l_dword     dst pack;
    l_byte      mbz pack;
    l_byte      protocol pack;
    l_ushort    length pack;
    l_ushort    checksum pack;
} tcp_PseudoHeader;

/*
 * TCP states, from tcp manual.
 * Note: close-wait state is bypassed by automatically closing a connection
 *       when a FIN is received.  This is easy to undo.
 */
#define tcp_StateLISTEN  0      /* listening for connection */
#define tcp_StateSYNSENT 1      /* syn sent, active open */
#define tcp_StateSYNREC  2      /* syn received, synack+syn sent. */
#define tcp_StateESTAB   3      /* established */
#define tcp_StateESTCL   4      /* established, but will FIN */
#define tcp_StateFINWT1  5      /* sent FIN */
#define tcp_StateFINWT2  6      /* sent FIN, received FINACK */
#define tcp_StateCLOSWT  7      /* received FIN waiting for close */
#define tcp_StateCLOSING 8      /* sent FIN, received FIN (waiting for FINACK) */
#define tcp_StateLASTACK 9      /* fin received, finack+fin sent */
#define tcp_StateTIMEWT  10     /* dally after sending final FINACK */
#define tcp_StateCLOSEMSL 11
#define tcp_StateCLOSED  12     /* finack received */

#define tcp_MaxBufSize 2048         /* maximum bytes to buffer on input */

/*
 * UDP socket definition
 */
typedef struct _udp_socket {
    struct _udp_socket *next;
    l_ushort        ip_type;            /* always set to UDP_PROTO */
    l_text          err_msg;            /* null when all is ok */
    l_text          usr_name;
    void          (*usr_yield)();
    l_ushort        sock_mode;          /* a logical OR of bits */
    l_dword         usertimer;          /* ip_timer_set, ip_timer_timeout */
    l_func          dataHandler;
    l_eth_address   hisethaddr;         /* peer's ethernet address */
    l_dword         hisaddr;            /* peer's internet address */
    l_ushort        hisport;            /* peer's UDP port */
    l_dword         myaddr;
    l_ushort        myport;
    l_ushort        locflags;

    l_int           queuelen;
    l_byte         *queue;

    l_int           rdatalen;           /* must be signed */
    l_ushort        maxrdatalen;
    l_byte         *rdata;
    l_byte          rddata[ tcp_MaxBufSize + 1];         /* if dataHandler = 0, len == 512 */
    l_dword         safetysig;

} udp_Socket;


/*
 * TCP Socket definition
*/
typedef struct _tcp_socket {

    struct _tcp_socket *next;
    l_ushort        ip_type;        /* always set to TCP_PROTO */
    l_char         *err_msg;
    l_char         *usr_name;
    void          (*usr_yield)();
    l_ushort        sock_mode;      /* a logical OR of bits */

    l_dword         usertimer;      /* ip_timer_set, ip_timer_timeout */
    l_func          dataHandler;    /* called with incoming data */
    l_eth_address   hisethaddr;     /* ethernet address of peer */
    l_dword         hisaddr;        /* internet address of peer */
    l_ushort        hisport;        /* tcp ports for this connection */
    l_dword         myaddr;
    l_ushort        myport;
    l_ushort        locflags;

    l_int           queuelen;
    l_byte         *queue;

    l_int           rdatalen;       /* must be signed */
    l_ushort        maxrdatalen;
    l_byte         *rdata;
    l_byte          rddata[tcp_MaxBufSize+1];    /* received data */
    l_dword         safetysig;
    l_word          state;          /* connection state */

    l_dword         acknum;
    l_dword         seqnum;         /* data ack'd and sequence num */
    l_long          timeout;        /* timeout, in milliseconds */
    l_byte          unhappy;        /* flag, indicates retransmitting segt's */
    l_byte          recent;         /* 1 if recently transmitted */
    l_ushort        flags;          /* tcp flags word for last packet sent */

    l_ushort        window;         /* other guy's window */
    l_int           datalen;        /* number of bytes of data to send */
				    /* must be signed */
    l_int           unacked;        /* unacked data */

    l_byte          cwindow;        /* Van Jacobson's algorithm */
    l_byte          wwindow;

    l_word          vj_sa;          /* VJ's alg, standard average */
    l_word          vj_sd;          /* VJ's alg, standard deviation */
    l_dword         vj_last;        /* last transmit time */
    l_word          rto;
    l_byte          karn_count;     /* count of packets */
    l_byte          tos;            /* priority */
    /* retransmission timeout proceedure */
    /* these are in clock ticks */
    l_dword         rtt_lasttran;       /* last transmission time */
    l_dword         rtt_smooth;         /* smoothed round trip time */
    l_dword         rtt_delay;          /* delay for next transmission */
    l_dword         rtt_time;           /* time of next transmission */

    l_ushort        mss;
    l_dword         inactive_to;           /* for the inactive flag */
    l_int           sock_delay;

    l_byte          data[tcp_MaxBufSize+1]; /* data to send */
} tcp_Socket;


/*
 * ARP definitions
 */
#define arp_TypeEther  0x100            /* ARP type of Ethernet address */

/* harp op codes */
#define ARP_REQUEST 0x0100
#define ARP_REPLY   0x0200

/*
 * Arp header
 */
typedef struct {
    l_ushort        hwType pack;
    l_ushort        protType pack;
    l_ushort        hwProtAddrLen pack;  /* hw and prot addr len */
    l_ushort        opcode pack;
    l_eth_address   srcEthAddr pack;
    l_dword         srcIPAddr pack;
    l_eth_address   dstEthAddr pack;
    l_dword         dstIPAddr pack;
} arp_Header;

typedef struct sockaddr_wat {
    l_ushort    s_type pack;
    l_ushort    s_port pack;
    l_dword     s_ip pack;
    l_byte      s_spares[6] pack;    /* unused in TCP realm */
} sockaddr_wat;


/**
 * extsock.h
*/
/* sock_type used for socket io */
typedef union {
    udp_Socket udp;
    tcp_Socket tcp;
} sock_type;

/*
 *  94.04.20 by MF
 *  fds - file descriptor
 *  lds - listen file descriptor
 */
typedef struct {
  sock_type  s;
  l_ushort   type;
  l_ushort   port;
  l_int fds, lds;
} ext_socket;

#define SYSSOCK 1
/**
*/


/**
 *
*/
extern l_ushort tcpip_sock_delay;
extern l_dword  tcpip_my_ip_addr;
extern l_int    tcpip_last_cookie;
extern l_dword  tcpip_cookie[MAX_COOKIES];

extern l_text  tcpip_def_domain;
extern l_dword tcpip_def_nameservers[MAX_NAMESERVERS];
extern l_int   tcpip_last_nameserver;
extern l_text  tcpip_hostname;

/**
 * from elib.h
*/
l_timestamp tcpip_set_timeout ( l_ushort seconds );
l_timestamp tcpip_set_ttimeout ( l_dword ticks );
l_bool      tcpip_chk_timeout ( l_timestamp timeout );


#define tcpip_intel     low_math_ntohl
#define tcpip_intel16   low_math_ntohs
#define tcpip_inchksum  low_math_inchksum
#define tcpip_checksum  tcpip_inchksum
/**
*/

l_int tcpip_ip_delay2( sock_type *s, l_int timeoutseconds, l_func fn, l_int *statusptr );
l_int tcpip_ip_delay1( sock_type *s, l_int timeoutseconds, l_func fn, l_int *statusptr );
l_int tcpip_ip_delay0( sock_type *s, l_int timeoutseconds, l_func fn, l_int *statusptr );

#define tcpip_sock_wait_established( s, seconds, fn, statusptr ) \
    if (tcpip_ip_delay0( s, seconds, fn, statusptr )) goto sock_err;
#define tcpip_sock_wait_input( s, seconds, fn , statusptr ) \
    if (tcpip_ip_delay1( s, seconds, fn, statusptr )) goto sock_err;
#define tcpip_sock_wait_closed(s, seconds, fn, statusptr )\
    if (tcpip_ip_delay2( s, seconds, fn, statusptr )) goto sock_err;
#define tcpip_sock_tick( s, statusptr ) \
    if ( !tcpip_tcp_tick(s)) { if (statusptr) *statusptr = 1 ; goto sock_err; }


/**
 * udp_nds.h
*/
/* Tests, if address @addr is correct IP address */
l_ushort tcpip_isaddr ( l_text addr );
/* Converts IP address from string to l_dword */
l_dword  tcpip_aton ( l_text addr );
/**
*/


/**
 * pcbsd.h
*/

/** Tests, if socket @s is real socket or not. ZERO if not,
 * 1 - if UDP socket, 2 if TCP socket
 */
l_int tcpip_chk_socket ( tcp_Socket *s );

/** Converts IP address in @x to string buffer @s and returns this buffer.
*/
l_text tcpip_inet_ntoa( l_text s, l_dword x );

/**
 * Converts IP address from @s to l_dword or return 0, if @s is not an IP
 * address
*/
l_dword tcpip_inet_addr ( l_text s );

/**
 * Returns IP address of this host.
*/
l_dword tcpip_gethostid ( void );

/**
 * Sets IP address of this host and returns it.
*/
l_dword tcpip_sethostid ( l_dword ip );

l_ushort tcpip_ntohs( l_ushort a );
l_ushort tcpip_htons ( l_ushort a );
l_dword  tcpip_ntohl ( l_dword x );
l_dword  tcpip_htonl ( l_dword x );

/**
*/


/**
 * lookup.h
*/
l_dword tcpip_lookup_host ( cl_text host, l_text ip_str );
/**
*/


/**
 * udp_dom.h
*/
/**
 * Converts domain name -> address resolution.
 * Returns 0 if name is unresolvable right now
 */
l_dword tcpip_resolve ( l_text name );

/**
*/


/**
 * netaddr.h
*/
l_dword tcpip_inet_aton ( cl_text name, struct in_addr *adr);

/*
 * Returns the network number from an internet
 * address; handles class A/B/C network #'s.
 */
l_dword tcpip_inet_netof (struct in_addr adr);

/*
 * Return the local network address portion of an
 * internet address; handles class A/B/C network
 * number formats only.
 * NB! return value is host-order
 */
l_dword tcpip_inet_lnaof (struct in_addr adr);
/**
*/


/**
 * pcarp.h
*/

extern l_ushort  tcpip_arp_last_gateway;

/*
 * tcpip_arp_add_gateway - if data is NULL, don't use string
 */
void  tcpip_arp_add_gateway ( l_text data, l_dword ip );
void  tcpip_arp_register( l_dword use, l_dword instead_of );
void  tcpip_arp_tick( l_dword ip );
l_int tcpip_arp_handler ( arp_Header *in );
l_int tcpip_arp_resolve( l_dword ina, l_eth_address *ethap, l_int nowait );
/**
*/

extern l_ushort tcpip_icmp_id;

/**
 * pctcp.h
*/
extern l_dword  tcpip_sin_mask;
extern l_ushort tcpip_mss;

void     tcpip_ip_timer_init( udp_Socket *s , l_int delayseconds );
l_int    tcpip_ip_timer_expired( udp_Socket *s );

l_text   tcpip_rip ( l_text s );
l_int    tcpip_sock_close( sock_type *s );
l_int    tcpip_sock_established( sock_type *s );
l_ushort tcpip_sock_dataready ( sock_type *s );
l_int    tcpip_sock_gets ( sock_type *s, l_byte *dp, l_int n );
l_int    tcpip_sock_puts ( sock_type *s, l_byte *dp );
l_ushort tcpip_sock_getc ( sock_type *s );
l_byte   tcpip_sock_putc( sock_type *s, l_byte c );
void     tcpip_sock_flushnext( sock_type *s);
void     tcpip_sock_flush( sock_type *s );
void     tcpip_sock_noflush( sock_type *s );
l_int    tcpip_sock_enqueue ( sock_type *s, l_byte *dp, l_int len );
l_int    tcpip_sock_setbuf ( sock_type *s, l_byte *dp, l_int len );
l_int    tcpip_sock_fastwrite(sock_type *s, l_byte *dp, l_int len);
l_int    tcpip_sock_write ( sock_type *s, l_byte *dp, l_int len );
l_int    tcpip_sock_fastread ( sock_type *s, l_byte *dp, l_int len );
l_int    tcpip_sock_read ( sock_type *s, l_byte *dp, l_int len );
void     tcpip_tcp_cancel ( in_Header *ip, l_int code, l_text msg, l_dword dummyip );
l_ptr    tcpip_tcp_lookup ( l_dword hisip, l_ushort hisport, l_ushort myport );
void     tcpip_udp_cancel( in_Header *ip );
l_int    tcpip_tcp_established(tcp_Socket *s);
l_int    tcpip_tcp_tick ( sock_type *s );
void     tcpip_tcp_sendsoon ( tcp_Socket *s );
void     tcpip_sock_abort ( tcp_Socket *s );
l_int    tcpip_tcp_listen(tcp_Socket *s, l_ushort lport, l_dword ina, l_ushort port, l_func datahandler, l_ushort timeout );
l_int    tcpip_tcp_open ( tcp_Socket *s, l_ushort lport, l_dword ina, l_ushort port, l_func datahandler );
l_int    tcpip_udp_open ( udp_Socket *s, l_ushort lport, l_dword ina, l_ushort port, l_func datahandler );
l_int    tcpip_tcp_init();
void     tcpip_tcp_shutdown();
l_int    tcpip_tcp_checkfor( tcp_Socket *t );

l_ushort tcpip_findfreeport ( l_ushort oldport );
l_int    tcpip_udp_close(udp_Socket *ds);
void     tcpip_tcp_close ( tcp_Socket *s );
void     tcpip_tcp_abort ( tcp_Socket *s );
l_int    tcpip_udp_write( udp_Socket *s, l_byte *datap, l_int len);
l_int    tcpip_udp_read ( udp_Socket *s, l_byte *datap, l_int maxlen );
l_int    tcpip_tcp_read ( tcp_Socket *s, l_byte *datap, l_int maxlen );
l_int    tcpip_tcp_write ( tcp_Socket *s, l_byte *dp, l_int len );
void     tcpip_tcp_Flush ( tcp_Socket *s );
l_int    tcpip_tcp_send ( tcp_Socket *s, l_int line );
l_int    tcpip_tcp_rst ( in_Header *his_ip, tcp_Header *oldtcpp );

/**
*/


/**
 * bootp.h
*/
extern l_dword  tcpip_bootphost;
extern l_ushort tcpip_bootptimeout;
extern l_ushort tcpip_bootpon;
/**
*/



/**
 * pcbuf.h
*/
l_int tcpip_sock_rbsize( tcp_Socket *s );
l_int tcpip_sock_rbused ( tcp_Socket *s );
l_int tcpip_sock_rbleft( tcp_Socket *s );
l_int tcpip_sock_tbsize( tcp_Socket *s );
l_int tcpip_sock_tbused( tcp_Socket *s );
l_int tcpip_sock_tbleft( tcp_Socket *s );
l_int tcpip_sock_preread ( sock_type *s, l_byte *dp, l_int len );
/**
*/

/**
 * pcping.h
*/
l_int tcpip_ping ( l_dword host, l_dword countnum );
/**
*/


/**
 * pcsed.h
*/
extern l_eth_address tcpip_eth_addr;		/* local ethernet address */
extern l_eth_address tcpip_eth_brdcast;	/* Ethernet broadcast address */
extern l_ushort tcpip_pktdevclass;	/* Ethernet = 1, SLIP = 6 */


/*
 *  Initialize the Ethernet Interface, and this package.  Enable input on
 *  all packet buffers.
 */
l_int tcpip_eth_init();
/*
 * _eth_FormatPacket places the next packet into the buffer and uses the
 * type field for protocol determination.  Note, I only maintain a single
 * output buffer, and it gets used quickly then released.  The benefits of
 * non-blocking systems are immense.
 */
l_byte *tcpip_eth_formatpacket( l_ptr eth_dest, l_ushort eth_type );
/*
 * _eth_send does the actual transmission once we are complete with the
 * buffer.  Do any last minute patches here, like fix the size.
 */
l_int tcpip_eth_send ( l_ushort len );
/*
 * _eth_free - free an input buffer once it is no longer needed
 * If pointer to NULL, release all buffers
 */
void tcpip_eth_free( l_ptr buf );
/*
 * _eth_arrived - if a new packet has arrived, read it and fill pointer
 * with type of packet
 */

l_byte *tcpip_eth_arrived ( l_ushort *type_ptr );
/*
 * _eth_release - release the hardware
 */
void tcpip_eth_release();
/*
 * _eth_hardware - return pointer to hardware address of a packet
 */
l_ptr tcpip_eth_hardware( l_byte *p );
/**
*/

/**
 * pcconfig.h
*/
extern l_ushort tcpip_sock_delay;
extern l_ushort tcpip_sock_inactive;  /* defaults to forever */
extern l_ushort tcpip_multihomes;
extern l_char   tcpip_defaultdomain[MAX_DOMAIN_LENGTH ];
extern l_dword  tcpip_cookie[MAX_COOKIES];
extern l_int    tcpip_last_cookie;
extern l_ushort tcpip_domaintimeout;

extern void     (*tcpip_usr_init)();

void  tcpip_add_server( l_int *counter, l_int max, l_dword *array, l_dword value );
l_int tcpip_tcp_config ( l_dword myip, l_dword netmask, l_dword remip, l_text pv );
/**
*/

/**
 * pcrecv.h
*/
l_int tcpip_sock_recv_init ( udp_Socket *s, l_ptr space, l_ushort len );
l_int tcpip_sock_recv_from( udp_Socket *s, l_long *hisip, l_ushort *hisport, l_text buffer, l_int len, l_ushort flags );
l_int tcpip_sock_recv( udp_Socket *s, l_text buffer, l_int len, l_ushort flags );
/**
*/

/**
 * wattcpd.h
*/
l_int tcpip_add_deamon ( void (*p)() );
l_int tcpip_delwattcpd( void (*p)() );
/**
*/


/**
 * fragment.h
*/
extern l_int active_frags;

/* Fragment is called if the frag section of
   the IP header is not zero or Frag Bit is set
   and Don't frag bit is not set */

l_byte *tcpip_fragment ( in_Header *ip );

void    tcpip_timeout_frags ( void );

/**
*/

/**
 * tcppkt.h
*/
extern l_int tcpip_pktipofs;

l_int tcpip_pkt_is_open ( void );
l_int tcpip_pkt_send ( l_ptr ptr, l_int len );
void  tcpip_pkt_buf_release ( l_text ptr );
void  tcpip_pkt_release ( void );
l_ptr tcpip_pkt_received ( void );
void  tcpip_pkt_buf_wipe ( void );
void  tcpip_pkt_buf_select ( l_byte *buf );
/**
*/


/**
 * tcpadd.h
*/
void   tcpip_base64encode ( l_text in, l_text out);
/**
*/

#ifdef __cplusplus
} /* end of poor C code */
#endif

#endif

